package Bencher::Scenario::HumanDateParsingModules::Parsing;

our $DATE = '2017-06-09'; # DATE
our $VERSION = '0.007'; # VERSION

use 5.010001;
use strict;
use warnings;

our $scenario = {
    summary => 'Benchmark human date parsing modules e.g. DF:Natural, DF:Alami, etc.',
    modules => {
        'DateTime::Format::Alami::EN' => {version => 0.13},
        'DateTime::Format::Alami::ID' => {version => 0.13},
    },
    participants => [
        {
            module=>'DateTime::Format::Alami::EN',
            code_template => 'state $parser = DateTime::Format::Alami::EN->new; $parser->parse_datetime(<text>)',
            tags => ['lang:en'],
        },
        {
            module=>'DateTime::Format::Alami::ID',
            code_template => 'state $parser = DateTime::Format::Alami::ID->new; $parser->parse_datetime(<text>)',
            tags => ['lang:id'],
        },
        {
            module=>'Date::Extract',
            code_template => 'state $parser = Date::Extract->new; $parser->extract(<text>)',
            tags => ['lang:en'],
        },
        {
            module=>'DateTime::Format::Natural',
            code_template => 'state $parser = DateTime::Format::Natural->new; $parser->parse_datetime(<text>)',
            tags => ['lang:en'],
        },
        {
            module=>'DateTime::Format::Flexible',
            code_template => 'DateTime::Format::Flexible->parse_datetime(<text>)',
            tags => ['lang:en'],
        },
    ],
    datasets => [
        {args => {text => '18 feb'}},
        {args => {text => '18 feb 2011'}},
        {args => {text => '18 feb 2011 06:30:45'}},
        {args => {text => 'today'}, include_participant_tags => ['lang:en']},
        {args => {text => 'hari ini'}, include_participant_tags => ['lang:id']},
    ],
};

1;
# ABSTRACT: Benchmark human date parsing modules e.g. DF:Natural, DF:Alami, etc.

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::HumanDateParsingModules::Parsing - Benchmark human date parsing modules e.g. DF:Natural, DF:Alami, etc.

=head1 VERSION

This document describes version 0.007 of Bencher::Scenario::HumanDateParsingModules::Parsing (from Perl distribution Bencher-Scenarios-HumanDateParsingModules), released on 2017-06-09.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m HumanDateParsingModules::Parsing

To run module startup overhead benchmark:

 % bencher --module-startup -m HumanDateParsingModules::Parsing

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<Date::Extract> 0.05.01

L<DateTime::Format::Alami::EN> 0.14

L<DateTime::Format::Alami::ID> 0.14

L<DateTime::Format::Flexible> 0.26

L<DateTime::Format::Natural> 1.04

=head1 BENCHMARK PARTICIPANTS

=over

=item * DateTime::Format::Alami::EN (perl_code) [lang:en]

Code template:

 state $parser = DateTime::Format::Alami::EN->new; $parser->parse_datetime(<text>)



=item * DateTime::Format::Alami::ID (perl_code) [lang:id]

Code template:

 state $parser = DateTime::Format::Alami::ID->new; $parser->parse_datetime(<text>)



=item * Date::Extract (perl_code) [lang:en]

Code template:

 state $parser = Date::Extract->new; $parser->extract(<text>)



=item * DateTime::Format::Natural (perl_code) [lang:en]

Code template:

 state $parser = DateTime::Format::Natural->new; $parser->parse_datetime(<text>)



=item * DateTime::Format::Flexible (perl_code) [lang:en]

Code template:

 DateTime::Format::Flexible->parse_datetime(<text>)



=back

=head1 BENCHMARK DATASETS

=over

=item * 18 feb

=item * 18 feb 2011

=item * 18 feb 2011 06:30:45

=item * today

=item * hari ini

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.24.0 >>, CPU: I<< Intel(R) Core(TM) i5-2400 CPU @ 3.10GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.0 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark with default options (C<< bencher -m HumanDateParsingModules::Parsing >>):

 #table1#
 {dataset=>"18 feb"}
 +-----------------------------+---------+-----------+-----------+------------+---------+---------+
 | participant                 | p_tags  | rate (/s) | time (ms) | vs_slowest |  errors | samples |
 +-----------------------------+---------+-----------+-----------+------------+---------+---------+
 | DateTime::Format::Flexible  | lang:en |       340 |      2.9  |        1   | 2.4e-05 |      21 |
 | Date::Extract               | lang:en |       600 |      2    |        2   |   2e-05 |      20 |
 | DateTime::Format::Alami::EN | lang:en |       960 |      1    |        2.8 | 3.6e-06 |      20 |
 | DateTime::Format::Natural   | lang:en |      1000 |      0.98 |        3   | 2.9e-06 |      20 |
 | DateTime::Format::Alami::ID | lang:id |      1500 |      0.68 |        4.3 | 1.5e-06 |      20 |
 +-----------------------------+---------+-----------+-----------+------------+---------+---------+

=begin html

<img src="data:image/png;base64,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" />

=end html


 #table2#
 {dataset=>"18 feb 2011"}
 +-----------------------------+---------+-----------+-----------+------------+---------+---------+
 | participant                 | p_tags  | rate (/s) | time (ms) | vs_slowest |  errors | samples |
 +-----------------------------+---------+-----------+-----------+------------+---------+---------+
 | Date::Extract               | lang:en |       180 |       5.6 |        1   | 1.8e-05 |      21 |
 | DateTime::Format::Natural   | lang:en |       210 |       4.7 |        1.2 | 1.7e-05 |      20 |
 | DateTime::Format::Flexible  | lang:en |       370 |       2.7 |        2.1 | 1.6e-05 |      21 |
 | DateTime::Format::Alami::EN | lang:en |       370 |       2.7 |        2.1 | 7.2e-06 |      21 |
 | DateTime::Format::Alami::ID | lang:id |       630 |       1.6 |        3.6 | 8.5e-06 |      21 |
 +-----------------------------+---------+-----------+-----------+------------+---------+---------+

=begin html

<img src="data:image/png;base64,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" />

=end html


 #table3#
 {dataset=>"18 feb 2011 06:30:45"}
 +-----------------------------+---------+-----------+-----------+------------+---------+---------+
 | participant                 | p_tags  | rate (/s) | time (ms) | vs_slowest |  errors | samples |
 +-----------------------------+---------+-----------+-----------+------------+---------+---------+
 | Date::Extract               | lang:en |       180 |       5.5 |        1   | 1.7e-05 |      20 |
 | DateTime::Format::Flexible  | lang:en |       350 |       2.8 |        1.9 | 1.8e-05 |      24 |
 | DateTime::Format::Natural   | lang:en |       440 |       2.3 |        2.4 | 1.1e-05 |      20 |
 | DateTime::Format::Alami::EN | lang:en |      1400 |       0.7 |        7.7 | 1.1e-06 |      21 |
 | DateTime::Format::Alami::ID | lang:id |      1400 |       0.7 |        7.7 |   2e-06 |      20 |
 +-----------------------------+---------+-----------+-----------+------------+---------+---------+

=begin html

<img src="data:image/png;base64,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" />

=end html


 #table4#
 {dataset=>"hari ini"}
 +-----------------------------+----------+---------+---------+------+-----------+-----------+------------+---------+---------+
 | participant                 | dataset  | ds_tags | p_tags  | perl | rate (/s) | time (μs) | vs_slowest |  errors | samples |
 +-----------------------------+----------+---------+---------+------+-----------+-----------+------------+---------+---------+
 | DateTime::Format::Alami::ID | hari ini |         | lang:id | perl |      3600 |       280 |          1 | 6.9e-07 |      20 |
 +-----------------------------+----------+---------+---------+------+-----------+-----------+------------+---------+---------+

=begin html

<img src="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAtAAAAH4CAMAAABUnipoAAAJJGlDQ1BpY2MAAHjalZVnUJNZF8fv8zzphUASQodQQ5EqJYCUEFoo0quoQOidUEVsiLgCK4qINEUQUUDBVSmyVkSxsCgoYkE3yCKgrBtXERWUF/Sd0Xnf2Q/7n7n3/OY/Z+4995wPFwCCOFgSvLQnJqULvJ3smIFBwUzwg8L4aSkcT0838I96Pwyg5XhvBfj3IkREpvGX4sLSyuWnCNIBgLKXWDMrPWWZDy8xPTz+K59dZsFSgUt8Y5mjv/Ho15xvLPqa4+vNXXoVCgAcKfoHDv+B/3vvslQ4gvTYqMhspk9yVHpWmCCSmbbcCR6Xy/QUJEfFJkT+UPC/Sv4HpUdmpy9HbnLKBkFsdEw68/8ONTIwNATfZ/HW62uPIUb//85nWd+95HoA2LMAIHu+e+GVAHTuAED68XdPbamvlHwAOu7wMwSZ3zzU8oYGBEABdCADFIEq0AS6wAiYAUtgCxyAC/AAviAIrAN8EAMSgQBkgVywDRSAIrAH7AdVoBY0gCbQCk6DTnAeXAHXwW1wFwyDJ0AIJsArIALvwTwEQViIDNEgGUgJUod0ICOIDVlDDpAb5A0FQaFQNJQEZUC50HaoCCqFqqA6qAn6BToHXYFuQoPQI2gMmob+hj7BCEyC6bACrAHrw2yYA7vCvvBaOBpOhXPgfHg3XAHXwyfgDvgKfBsehoXwK3gWAQgRYSDKiC7CRriIBxKMRCECZDNSiJQj9Ugr0o30IfcQITKDfERhUDQUE6WLskQ5o/xQfFQqajOqGFWFOo7qQPWi7qHGUCLUFzQZLY/WQVugeehAdDQ6C12ALkc3otvR19DD6An0ewwGw8CwMGYYZ0wQJg6zEVOMOYhpw1zGDGLGMbNYLFYGq4O1wnpgw7Dp2AJsJfYE9hJ2CDuB/YAj4pRwRjhHXDAuCZeHK8c14y7ihnCTuHm8OF4db4H3wEfgN+BL8A34bvwd/AR+niBBYBGsCL6EOMI2QgWhlXCNMEp4SyQSVYjmRC9iLHErsYJ4iniDOEb8SKKStElcUggpg7SbdIx0mfSI9JZMJmuQbcnB5HTybnIT+Sr5GfmDGE1MT4wnFiG2RaxarENsSOw1BU9Rp3Ao6yg5lHLKGcodyow4XlxDnCseJr5ZvFr8nPiI+KwETcJQwkMiUaJYolnipsQUFUvVoDpQI6j51CPUq9RxGkJTpXFpfNp2WgPtGm2CjqGz6Dx6HL2IfpI+QBdJUiWNJf0lsyWrJS9IChkIQ4PBYyQwShinGQ8Yn6QUpDhSkVK7pFqlhqTmpOWkbaUjpQul26SHpT/JMGUcZOJl9sp0yjyVRclqy3rJZskekr0mOyNHl7OU48sVyp2WeywPy2vLe8tvlD8i3y8/q6Co4KSQolCpcFVhRpGhaKsYp1imeFFxWommZK0Uq1SmdEnpJVOSyWEmMCuYvUyRsryys3KGcp3ygPK8CkvFTyVPpU3lqSpBla0apVqm2qMqUlNSc1fLVWtRe6yOV2erx6gfUO9Tn9NgaQRo7NTo1JhiSbN4rBxWC2tUk6xpo5mqWa95XwujxdaK1zqodVcb1jbRjtGu1r6jA+uY6sTqHNQZXIFeYb4iaUX9ihFdki5HN1O3RXdMj6Hnppen16n3Wl9NP1h/r36f/hcDE4MEgwaDJ4ZUQxfDPMNuw7+NtI34RtVG91eSVzqu3LKya+UbYx3jSONDxg9NaCbuJjtNekw+m5qZCkxbTafN1MxCzWrMRth0tie7mH3DHG1uZ77F/Lz5RwtTi3SL0xZ/Wepaxls2W06tYq2KXNWwatxKxSrMqs5KaM20DrU+bC20UbYJs6m3eW6rahth22g7ydHixHFOcF7bGdgJ7Nrt5rgW3E3cy/aIvZN9of2AA9XBz6HK4ZmjimO0Y4ujyMnEaaPTZWe0s6vzXucRngKPz2viiVzMXDa59LqSXH1cq1yfu2m7Cdy63WF3F/d97qOr1Vcnre70AB48j30eTz1Znqmev3phvDy9qr1eeBt653r3+dB81vs0+7z3tfMt8X3ip+mX4dfjT/EP8W/ynwuwDygNEAbqB24KvB0kGxQb1BWMDfYPbgyeXeOwZv+aiRCTkIKQB2tZa7PX3lwnuy5h3YX1lPVh68+EokMDQptDF8I8wurDZsN54TXhIj6Xf4D/KsI2oixiOtIqsjRyMsoqqjRqKtoqel/0dIxNTHnMTCw3tir2TZxzXG3cXLxH/LH4xYSAhLZEXGJo4rkkalJ8Um+yYnJ28mCKTkpBijDVInV/qkjgKmhMg9LWpnWl05c+xf4MzYwdGWOZ1pnVmR+y/LPOZEtkJ2X3b9DesGvDZI5jztGNqI38jT25yrnbcsc2cTbVbYY2h2/u2aK6JX/LxFanrce3EbbFb/stzyCvNO/d9oDt3fkK+Vvzx3c47WgpECsQFIzstNxZ+xPqp9ifBnat3FW560thROGtIoOi8qKFYn7xrZ8Nf674eXF31O6BEtOSQ3swe5L2PNhrs/d4qURpTun4Pvd9HWXMssKyd/vX779Zblxee4BwIOOAsMKtoqtSrXJP5UJVTNVwtV11W418za6auYMRB4cO2R5qrVWoLar9dDj28MM6p7qOeo368iOYI5lHXjT4N/QdZR9tapRtLGr8fCzpmPC49/HeJrOmpmb55pIWuCWjZfpEyIm7J+1PdrXqtta1MdqKToFTGade/hL6y4PTrqd7zrDPtJ5VP1vTTmsv7IA6NnSIOmM6hV1BXYPnXM71dFt2t/+q9+ux88rnqy9IXii5SLiYf3HxUs6l2cspl2euRF8Z71nf8+Rq4NX7vV69A9dcr9247nj9ah+n79INqxvnb1rcPHeLfavztuntjn6T/vbfTH5rHzAd6Lhjdqfrrvnd7sFVgxeHbIau3LO/d/0+7/7t4dXDgw/8HjwcCRkRPox4OPUo4dGbx5mP559sHUWPFj4Vf1r+TP5Z/e9av7cJTYUXxuzH+p/7PH8yzh9/9UfaHwsT+S/IL8onlSabpoymzk87Tt99ueblxKuUV/MzBX9K/FnzWvP12b9s/+oXBYom3gjeLP5d/Fbm7bF3xu96Zj1nn71PfD8/V/hB5sPxj+yPfZ8CPk3OZy1gFyo+a33u/uL6ZXQxcXHxPy6ikLxyKdSVAAAAIGNIUk0AAHomAACAhAAA+gAAAIDoAAB1MAAA6mAAADqYAAAXcJy6UTwAAAB7UExURf///wAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAP8AAP8AAP8AAAAAAAAAAAAAAP8AAAAAAMQAAJ4AADoAAAoAACYAAAAAAP8AAHcAAP///2QVSJQAAAAldFJOUwARRGYid7sz7t2ZiMyqVY515NXHP/b87BF1RKPw+SD0mb7wytWDpakzAAAAAWJLR0QAiAUdSAAAAAlwSFlzAAAASAAAAEgARslrPgAAAAd0SU1FB+EGCQogGob1gKMAABO2SURBVHja7d0Ld6M4moBhITCgi5lU9dRsd2/Nzt40+///4UrC2E422XI+XM4X1fucKWMIcZLT72EkfMEYAAAAAAAAAAAAAAAAAAAAAD9fZ093bPfRvwog1Q/bPZtOd5K93uEwfvSvCNxuPNf7etDTTND4PHrnc759CDYHPZTFGrQNYSpf72IkaHweNejRhTD3Ns3ep1CD7ufgU5+/HnwgaHwiZcgx5oNxONp0yIGnLgc9pLzlMBuzuI6g8ZmsY2i7xHEdQ+fDc7K9s1maunkyBI3PpAQd0ujH66DDPBZTiCHk8ciw94cAj5KDXuZcbMhBd3kSWINenKnnpfPUkKDxqYxLnhiWsxk5aG+Md+Ug3c15Qhhc3YEhBz6To+u76Ebn57/E6FweNK9nOWKc63k7gsan0tk8oLC2M9ty28wT4AAAAAAAAAAAAADwc01jmkN5bja45NfnaOubOTuft3/0Lwe8Uzf7YXG53BAXG33ZtL6Z00e7lFeKAZ9JfS16eRnYvOSVemiub+Ysr+bNkX/0rwe8T/1MFH8sYQ/re+7XN3PW0M9vxwc+jzHGKbd7TMlN5zdzLs+DfvpSff0N+ElcTcz9dXfQIeTZXzi9yWJ7M2f/POjfvhH0u3358tG/waeyBv23pzsco/PhuLY7pPObOV8MOX777YH/l9GKwFmi9/uXp33fX98WlMOdSrtdmrY3cw5pMPUtciuCFiBogb1B2/JBKOX9nO7w/M2cMf/XGM//RQhagKAF9gZtfBrr+zmnObrrN3OW9Xh+MxxBCxC0wO6gzWDX03Uv38zZ2auPziRoAYIW2B/0TQhawNr9j/HLIWg8hH3l3s9A0PjpyrUJLk8ap/9/x50IGj9d+RjKm47Ql4sY/P7HjX5/8RAEjXsa+m69+MB6LYI8szWHv5SPcg/rpvJhlK/t1IfDdhGD6s9/3ujPF78AQeOebDxdfGC9FkEeYPj4r6XTVJ6uCMd5yPde2am+foKgoc354gPrtQhyq2EdSZQnk4f6XFt6fad+vB5yEDRUOH9W+3otgvXuGvT24anp9Z0sQUOfc6vrtQh+EPT1TgQNhbaLD5yuRfAs6PL+JdvXoF/ZiaChUL34QJi3axGcgl5K0EO+27lDDfqVnWrQy/Y4BA0VbBqdmxdzuhZBX1s9unJcztPAsbxeLb2+Uwm67Lj6/c8bcR4aP1MeHp9epLZdi8Ccrk1gLq9je3Wn6x13IGjc003vi/6Zb54maNwTQQP3RNBoCkGjKQSNphA0mkLQaApBoykEjaYQNJpC0GgKQaMpBI2mEDSaQtBoCkGjKQSNphA0mkLQaApBoykEjaYQNJpC0GgKQaMpBI2mEDSaQtBoCkGjKQSNpuwOehrTHMpHowaXfHdZ73xenvciaDzG3qC72Q+Ly+WGuNjoL+s+2mXePr6aoPEge4Oun4xaLpMxL/VD2Lf1rnwoe72qRkXQeIzdR+jymez+WEIe7PP1688BJmg8xh0mhWOMU273mJKbzuvLi6C/2mL/tcmBN0w1se9Pux+oXtU2lAsbeXde718E/X0sevEPAX4g1MS+Pd3hoZbTtRSHND1bZ8iBR9s75KiXB7XlQs6mXEtx2taHcmHn3m27ETQeY/9ZjmkdarhDORV9WY+hXqn8hKDxGLsnhT6Nbs4RT3Osy+v12G17ETQeY/9Zju1qit3pWopX65edCBqPwWs50BSCRlMIGk0haDSFoNEUgkZTCBpNIWg0haDRFIJGUwgaTSFoNIWg0RSCRlMIGk0haDSFoNEUgkZTCBpNIWg0haDRFIJGUwgaTSFoNIWg0RSCRlMIGk0haDSFoNEUgkZTCBpNIWg0haDRFIJGUwgaTSFoNIWg0RSCRlMIGk0haDSFoNEUgkZTCBpN2R30NKY5lIt8B5d8Xg7HNJdl5/P2814EjcfYG3Q3+2FxudwQFxu9MfE4TXE0xke7zP22G0HjMfYGbVO+CTngeckro5lSOUqnqUu2RL7tRtB4jN1H6Nyt8ccS9lDuDmWYMaShhl5vKoLGY9xhUjjGOOV2jym5qW7o8pBjIWh8hDsEHUKe/YWUx8/elfXDPHamfxF0qoL4hwA/MNbE/v50h4fKh+Pabh4758NzLEMPhhz4EHuP0GU+WMKdSrtdDrqc6TB1GG1M77bdCBqPsf8sx7QONdyhnIrOB2tbGBPz8GI8DzEIGo+xewzt0+jmHPU0x7IM62B5XY/dthdB4zH2TwqHcjw25QSe7a63d6ftFUHjMXgtB5pC0GgKQaMpBI2mEDSaQtBoCkGjKQSNphA0mkLQaApBoykEjaYQNJpC0GgKQaMpBI2mEDSaQtBoCkGjKQSNphA0mkLQaApBoykEjaYQNJpC0GgKQaMpBI2mEDSaQtBoCkGjKQSNphA0mkLQaApBoykEjaYQNJpC0GgKQaMpBI2mEDSaQtBoCkGjKbuDnsY0h3KR7+CSrxf7HlK57Xzeft6LoPEYe4PuZj8sLpcb4mKjz1umYw3aR7vM/bYbQeMx9gZtS71hNGZe8kpehpTKpi7ZEvm2G0HjMXYfoXO3xh9L2INdN9XGLzcVQeMx7jApHGOccrvHlNxU1mvGC0HjI9wh6BDy7C+kPH72rqzXjPsXQX8fi176M4AfCTWxb093eKh8OK7tDqkcol8dcny1xfDRfzTaNdXEvj/te5QyHyzhTutM8Bz0kHK8vdt2Y8iBx9h/lmNahxruUE5Fr5vKbQx5dH0+EU3QeIzdY2ifRjfnqKc51uUWdFmP3bYXQeMx9k8KB7ueruus7a63d6ftFUHjMXgtB5pC0GgKQaMpBI2mEDSaQtBoCkGjKQSNphA0mkLQaApBoykEjaYQNJpC0GgKQaMpBI2mEDSaQtBoCkGjKQSNphA0mkLQaApBoykEjaYQNJpC0GgKQaMpBI2mEDSaQtBoCkGjKQSNphA0mkLQaApBoykEjaYQNJpyW9DTtPPHEDQe45ag+zmN1u1qmqDxGDcEPaWDHbswdz9+tDcRNB7jhqCDN3Y0JtofP9qbCBqPcUvQgaDxWdwQtJ2nHHTPkAOfwE2TwuRmN/d7fgxB4zFuOm039GF56/g8jWkO5YvBJZ+Xnc/r5rJcETQe44agh3Xw3A+vfbGb/bC4XG6Ii43eGB/tUo7m23JF0HiMHwY92IO32eJenRTalG9CnjPOiylzxy7ZEvd5eULQeIwfBt2P0Y3F8dVBR1cy98cSdj2S18DzzbY8IWg8xi1PrKwDh9eHHNkY45TbPabkJrOcQl5eBP0lFHvO/P16/u2Px/jov/NO+prYP55u2PNYjtDzWzWGkGd/IZXxszP9KeSeoHf745+P8dF/553cHLSdQxxDmfC9ZTkNMYY0MeS4H4IWuO2ZwsWbzr06hi7zwRLuVNrt0jSkPDLpndmWJwQtQNACtwU95WzHN85yTHWoYdyhnIo2Joa8a7gsVwQtQNACNwTd57lePt6+ftrO+DS6OUc9zfG8jN1luSJoAYIWuOWZwnE0YXbxja8Odi29s7Y7Lc31siJoAYIWuPUtWEu/57VJBC1B0AK3vNpu18uSVgQtQNACNwR9iD9+mB8haAGCFrhlyOFDeTHHridFCFqAoAVuGXKk1Z4fQ9ACBC3A53LoRdACBK0XQQsQtF4ELUDQehG0AEHrRdACBK0XQQsQtF4ELUDQehG0AEHrRdACBK0XQQsQtF4ELUDQehG0AEHrRdACBK0XQQsQtF4ELUDQehG0AEHrRdACBK0XQQsQtF4ELUDQehG0AEHrRdACBK0XQQsQtF4ELUDQehG0AEHrRdACBK0XQQsQtF4ELUDQehG0AEHrRdACBK0XQQsQtF4ELUDQehG0AEHrRdACBK0XQQsQtF4ELUDQehG0AEHrRdACBK0XQQvsDnoa0xw6Y0K58tuY1495PW/v/LpcEbQAQQvsDbqb/bC4XO7RW2sn0zk/LaVkH+0yny+qTNACBC2wN2hbrscZ8pF5XOPt53LjTJds3n6+qDJBCxC0wO4jdLlisj8ak/oQ7Np2qbyGbs9XnyVoAYIWuMOkcIxxykG74FNvljSU8bRdXgT9tV4ufPjoP/dTIeh3mWpi3592P1AIecw8lInhIY83RheOcxr6F0F/H4t+x4/59RD0u4Sa2LenOzzUcgq3Dpz7cFgYctwDQQvsHXKcx8zl4DukqY6p86RwKEOPvDwhaAGCFth/liOPn707LWM9SHflNF7M/8bziWiCFiBogd2TQp9GN09lIjg6l5d9Gmeft09zdLHb9iJoAYIW2H+WY7D2xXKqy+60XhG0AEEL8FoOvQhagKD1ImgBgtaLoAUIWi+CFiBovQhagKD1ImgBgtaLoAUIWi+CFiBovQhagKD1ImgBgtaLoAUIWi+CFiBovQhagKD1ImgBgtaLoAUIWi+CFiBovQhagKD1ImgBgtaLoAUIWi+CFiBovQhagKD1ImgBgtaLoAUIWi+CFiBovQhagKD1ImgBgtaLoAUIWi+CFiBovQhagKD1ImgBgtaLoAUIWi+CFiBovQhagKD1ImgBgtaLoAUIWi+CFiBovQhagKD1ImgBgtaLoAUIWi+CFiBovQhagKD1ImgBgtaLoAV2Bz2NaQ5duXh9NhozHNf1zufleS+CFiBogb1Bd7MfFpfLPXpry2XrY16WdR/tMvfbbgQtQNACe4O2Kd+EfGQeT/EmW9e7uozbbgQtQNACu4/QuVvjjznkPoRyfz7ko/VxDb3eVAQtQNACd5gUjjHmoUZywad8lJ7SGF1nFoLejaAF7hB0CHn2N5SJ4GHOY+hx6fMYun8RdKrCjh/z6yHodxlrYn9/usNDLadw88C5z03nkAeGHPsRtMDeI3SZD5ZwbZkTDmmq88Ac9pAGY3q37UbQAgQtsP8sRx4/e3daxjyEnk5Djzy8GM9DDIIWIGiB3WNon0Y3T+WJldG5EnOKdX2ao4vdthdBCxC0wP5J4WDty2XtuDutVwQtQNACvJZDL4IWIGi9CFqAoPUiaAGC1ougBQhaL4IWIGi9CFqAoPUiaAGC1ougBQhaL4IWIGi9CFqAoPUiaAGC1ougBQhaL4IWIGi9CFqAoPUiaAGC1ougBQhaL4IWIGi9CFqAoPUiaAGC1ougBQhaL4IWIGi9CFqAoPUiaAGC1ougBQhaL4IWIGi9CFqAoPUiaAGC1ougBQhaL4IWIGi9CFqAoPUiaAGC1ougBQhaL4IWIGi9CFqAoPUiaAGC1ougBQhaL4IWIGi9CFqAoPUiaAGC1ougBQhaL4IWIGi9CFqAoPUiaIHdQU9jmkNXLl6fjcamyprO5+3nvQhagKAF9gbdzX5YXC736K21U7lkvbWHuTM+2mXut90IWoCgBfYGbVO+CaMxY3/ZGBfT5YO0CXHbQtACBC2w+widuzX+aEzqQ7DrtsPxFHq9qQhagKAF7jApHGOcctAu+FSP0t08GLMQ9G4ELXCHoEPIs7+hTAwPc133+aZ/EfT3seilP+OXRNDvEmpi357u8FDLKdw6cO7mMvJ4OeT4WiaLdvjoP/pTIeh3mWpi35/2PUqZD5ZwbTn4DimPPXpn6t1hu1sw5BAgaIH9Zzlyw96dluWsxtHXL8SQR9fnE9EELUDQArvH0D6Nbp7KEyujc3lpTiefpzm62G17EbQAQQvsnxQO1j5bbrrrdYIWIGgBXsuhF0ELELReBC1A0HoRtABB60XQAgStF0ELELReBC1A0HoRtABB60XQAgStF0ELELReBC1A0HoRtABB60XQAgStF0ELELReBC1A0HoRtABB60XQAgStF0ELELReBC1A0HoRtABB60XQAgStF0ELELReBC1A0HoRtABB60XQAgStF0ELELReBC1A0HoRtABB60XQAgStF0ELELReBC1A0HoRtABB60XQAgStF0ELELReBC1A0HoRtABB60XQAgStF0ELELReBC1A0HoRtABB60XQAgStF0ELELReBC1A0Hr9O0G/H0Hr9R8E/X67g57GNIfOmJCyMW8ILvm83vm8/bwXQQsQtMDeoLvZD4vL5R69tXbKPcfFRm+Mj3aZ+203ghYgaIG9QduUb0I+Mo+neOclbxxNl2yJe9uNoAUIWmD3ETp3a/zRmNSHYGvgQ9lUQ683FUELELTAHSaFY4x5qJFc8KnPDR9TcpNZXgT97Uvx9Tfc7j//5zE++u+8E1cT+9v+oEPIs7+hTAwPc54blvGzM/3zoJ++EPS7/dd/P8ZH/513sgbt/nqHw/xyCjcPnGvDQ5peDDmAz6HMB0u4tswJc8hTabhL05AGY3r30b8e8D42TXWIsS6jMe5QTkUbE0MeXYf9PwB4KJ9GN5fzz3mZJ4NmmmNdL8vYffRvB7zXYO2zZWdtd1p+9K8GAAAAM1m7js/qnXp/2u6cdXa4/p5huvo+QJOxvEIxhe1OSna7dzX96NOzk5/1dTPb9wGa1NOadn2V7elJqP97pvM4ljOiZzXoq+8D1Fjr7edy+yJoG9Z/5cmqGK42nIM+fR+gxhpmfaXty6BDWv+Zgyv/u2y4BN0lzotClVO9aTGXoN2Y9Zd98uF5uC73EvT6fYAaW5jXR+jyxp+r8xpD6q11/vI910FzhIYqa5hDnfS9MSmsb9VMV6PlS9DDs8ki8OHWMEM9LfdG0G6dGF4GF5egA69lhC51fBHWWl8/yzGlOvqI/vlZjqvvA9SoT5DEtcvXz3L49W3Gh9Q9O8tx9X0AAAAAAAAAAADAnf0vIDUNPRru9bYAAAAldEVYdGRhdGU6Y3JlYXRlADIwMTctMDYtMDlUMTc6MzI6MjYrMDc6MDCeTj+uAAAAJXRFWHRkYXRlOm1vZGlmeQAyMDE3LTA2LTA5VDE3OjMyOjI2KzA3OjAw7xOHEgAAACF0RVh0cHM6SGlSZXNCb3VuZGluZ0JveAA1MDR4NzIwKzUwKzUw1uIiwwAAABN0RVh0cHM6TGV2ZWwAQWRvYmUtMi4wCjmTdA0AAAAASUVORK5CYII=" />

=end html


 #table5#
 {dataset=>"today"}
 +-----------------------------+-----------+-----------+------------+---------+---------+
 | participant                 | rate (/s) | time (ms) | vs_slowest |  errors | samples |
 +-----------------------------+-----------+-----------+------------+---------+---------+
 | DateTime::Format::Flexible  |       350 |     2.8   |       1    | 1.7e-05 |      21 |
 | Date::Extract               |       990 |     1     |       2.8  | 5.6e-06 |      20 |
 | DateTime::Format::Natural   |      2700 |     0.37  |       7.7  | 3.7e-07 |      20 |
 | DateTime::Format::Alami::EN |      3500 |     0.286 |       9.88 | 2.7e-07 |      20 |
 +-----------------------------+-----------+-----------+------------+---------+---------+

=begin html

<img src="data:image/png;base64,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" />

=end html


Benchmark module startup overhead (C<< bencher -m HumanDateParsingModules::Parsing --module-startup >>):

 #table6#
 +-----------------------------+------------------------------+--------------------+----------------+-----------+------------------------+------------+-----------+---------+
 | participant                 | proc_private_dirty_size (MB) | proc_rss_size (MB) | proc_size (MB) | time (ms) | mod_overhead_time (ms) | vs_slowest |  errors   | samples |
 +-----------------------------+------------------------------+--------------------+----------------+-----------+------------------------+------------+-----------+---------+
 | DateTime::Format::Flexible  | 0.82                         | 4.1                | 20             |     120   |                  114.2 |        1   |   0.00031 |      20 |
 | Date::Extract               | 16                           | 20                 | 60             |     110   |                  104.2 |        1.1 |   0.00016 |      20 |
 | DateTime::Format::Natural   | 16                           | 20                 | 60             |     110   |                  104.2 |        1.1 |   0.00014 |      20 |
 | DateTime::Format::Alami::ID | 16                           | 20                 | 60             |      25   |                   19.2 |        4.8 | 6.5e-05   |      20 |
 | DateTime::Format::Alami::EN | 2.8                          | 6.4                | 24             |      25   |                   19.2 |        4.9 | 6.3e-05   |      20 |
 | perl -e1 (baseline)         | 2.8                          | 6.3                | 24             |       5.8 |                    0   |       21   | 3.7e-05   |      20 |
 +-----------------------------+------------------------------+--------------------+----------------+-----------+------------------------+------------+-----------+---------+


=begin html

<img src="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAtAAAAH4CAMAAABUnipoAAAJJGlDQ1BpY2MAAHjalZVnUJNZF8fv8zzphUASQodQQ5EqJYCUEFoo0quoQOidUEVsiLgCK4qINEUQUUDBVSmyVkSxsCgoYkE3yCKgrBtXERWUF/Sd0Xnf2Q/7n7n3/OY/Z+4995wPFwCCOFgSvLQnJqULvJ3smIFBwUzwg8L4aSkcT0838I96Pwyg5XhvBfj3IkREpvGX4sLSyuWnCNIBgLKXWDMrPWWZDy8xPTz+K59dZsFSgUt8Y5mjv/Ho15xvLPqa4+vNXXoVCgAcKfoHDv+B/3vvslQ4gvTYqMhspk9yVHpWmCCSmbbcCR6Xy/QUJEfFJkT+UPC/Sv4HpUdmpy9HbnLKBkFsdEw68/8ONTIwNATfZ/HW62uPIUb//85nWd+95HoA2LMAIHu+e+GVAHTuAED68XdPbamvlHwAOu7wMwSZ3zzU8oYGBEABdCADFIEq0AS6wAiYAUtgCxyAC/AAviAIrAN8EAMSgQBkgVywDRSAIrAH7AdVoBY0gCbQCk6DTnAeXAHXwW1wFwyDJ0AIJsArIALvwTwEQViIDNEgGUgJUod0ICOIDVlDDpAb5A0FQaFQNJQEZUC50HaoCCqFqqA6qAn6BToHXYFuQoPQI2gMmob+hj7BCEyC6bACrAHrw2yYA7vCvvBaOBpOhXPgfHg3XAHXwyfgDvgKfBsehoXwK3gWAQgRYSDKiC7CRriIBxKMRCECZDNSiJQj9Ugr0o30IfcQITKDfERhUDQUE6WLskQ5o/xQfFQqajOqGFWFOo7qQPWi7qHGUCLUFzQZLY/WQVugeehAdDQ6C12ALkc3otvR19DD6An0ewwGw8CwMGYYZ0wQJg6zEVOMOYhpw1zGDGLGMbNYLFYGq4O1wnpgw7Dp2AJsJfYE9hJ2CDuB/YAj4pRwRjhHXDAuCZeHK8c14y7ihnCTuHm8OF4db4H3wEfgN+BL8A34bvwd/AR+niBBYBGsCL6EOMI2QgWhlXCNMEp4SyQSVYjmRC9iLHErsYJ4iniDOEb8SKKStElcUggpg7SbdIx0mfSI9JZMJmuQbcnB5HTybnIT+Sr5GfmDGE1MT4wnFiG2RaxarENsSOw1BU9Rp3Ao6yg5lHLKGcodyow4XlxDnCseJr5ZvFr8nPiI+KwETcJQwkMiUaJYolnipsQUFUvVoDpQI6j51CPUq9RxGkJTpXFpfNp2WgPtGm2CjqGz6Dx6HL2IfpI+QBdJUiWNJf0lsyWrJS9IChkIQ4PBYyQwShinGQ8Yn6QUpDhSkVK7pFqlhqTmpOWkbaUjpQul26SHpT/JMGUcZOJl9sp0yjyVRclqy3rJZskekr0mOyNHl7OU48sVyp2WeywPy2vLe8tvlD8i3y8/q6Co4KSQolCpcFVhRpGhaKsYp1imeFFxWommZK0Uq1SmdEnpJVOSyWEmMCuYvUyRsryys3KGcp3ygPK8CkvFTyVPpU3lqSpBla0apVqm2qMqUlNSc1fLVWtRe6yOV2erx6gfUO9Tn9NgaQRo7NTo1JhiSbN4rBxWC2tUk6xpo5mqWa95XwujxdaK1zqodVcb1jbRjtGu1r6jA+uY6sTqHNQZXIFeYb4iaUX9ihFdki5HN1O3RXdMj6Hnppen16n3Wl9NP1h/r36f/hcDE4MEgwaDJ4ZUQxfDPMNuw7+NtI34RtVG91eSVzqu3LKya+UbYx3jSONDxg9NaCbuJjtNekw+m5qZCkxbTafN1MxCzWrMRth0tie7mH3DHG1uZ77F/Lz5RwtTi3SL0xZ/Wepaxls2W06tYq2KXNWwatxKxSrMqs5KaM20DrU+bC20UbYJs6m3eW6rahth22g7ydHixHFOcF7bGdgJ7Nrt5rgW3E3cy/aIvZN9of2AA9XBz6HK4ZmjimO0Y4ujyMnEaaPTZWe0s6vzXucRngKPz2viiVzMXDa59LqSXH1cq1yfu2m7Cdy63WF3F/d97qOr1Vcnre70AB48j30eTz1Znqmev3phvDy9qr1eeBt653r3+dB81vs0+7z3tfMt8X3ip+mX4dfjT/EP8W/ynwuwDygNEAbqB24KvB0kGxQb1BWMDfYPbgyeXeOwZv+aiRCTkIKQB2tZa7PX3lwnuy5h3YX1lPVh68+EokMDQptDF8I8wurDZsN54TXhIj6Xf4D/KsI2oixiOtIqsjRyMsoqqjRqKtoqel/0dIxNTHnMTCw3tir2TZxzXG3cXLxH/LH4xYSAhLZEXGJo4rkkalJ8Um+yYnJ28mCKTkpBijDVInV/qkjgKmhMg9LWpnWl05c+xf4MzYwdGWOZ1pnVmR+y/LPOZEtkJ2X3b9DesGvDZI5jztGNqI38jT25yrnbcsc2cTbVbYY2h2/u2aK6JX/LxFanrce3EbbFb/stzyCvNO/d9oDt3fkK+Vvzx3c47WgpECsQFIzstNxZ+xPqp9ifBnat3FW560thROGtIoOi8qKFYn7xrZ8Nf674eXF31O6BEtOSQ3swe5L2PNhrs/d4qURpTun4Pvd9HWXMssKyd/vX779Zblxee4BwIOOAsMKtoqtSrXJP5UJVTNVwtV11W418za6auYMRB4cO2R5qrVWoLar9dDj28MM6p7qOeo368iOYI5lHXjT4N/QdZR9tapRtLGr8fCzpmPC49/HeJrOmpmb55pIWuCWjZfpEyIm7J+1PdrXqtta1MdqKToFTGade/hL6y4PTrqd7zrDPtJ5VP1vTTmsv7IA6NnSIOmM6hV1BXYPnXM71dFt2t/+q9+ux88rnqy9IXii5SLiYf3HxUs6l2cspl2euRF8Z71nf8+Rq4NX7vV69A9dcr9247nj9ah+n79INqxvnb1rcPHeLfavztuntjn6T/vbfTH5rHzAd6Lhjdqfrrvnd7sFVgxeHbIau3LO/d/0+7/7t4dXDgw/8HjwcCRkRPox4OPUo4dGbx5mP559sHUWPFj4Vf1r+TP5Z/e9av7cJTYUXxuzH+p/7PH8yzh9/9UfaHwsT+S/IL8onlSabpoymzk87Tt99ueblxKuUV/MzBX9K/FnzWvP12b9s/+oXBYom3gjeLP5d/Fbm7bF3xu96Zj1nn71PfD8/V/hB5sPxj+yPfZ8CPk3OZy1gFyo+a33u/uL6ZXQxcXHxPy6ikLxyKdSVAAAAIGNIUk0AAHomAACAhAAA+gAAAIDoAAB1MAAA6mAAADqYAAAXcJy6UTwAAADhUExURf///wAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAoAACYAAAAAACIAAAAAAAAAAAAAAGgAAP8AAAAAAJQAAP8AAP8AAAAAAAAAAP8AAP8AAAAAAAAAAAAAAAAAAAAAAP8AAP8AAP8AAP8AAP8AAP8AAP8AAP8AAN8AAMQAAO0AANQAAPIAAJ4AALkAAOcAAGIAADoAAFAAAGsAAGsAAD8AAB0AABMAAAAAAHcAAP8AAMcAAN8AAF0AAGgAAP///71LmQ8AAABDdFJOUwARRGYiiN2Zu8xVM+6qd3BAXNXkx8rVP+vw/HWnROx1EXWOo8/H+fb0aU63jnrfp2kg4+SZ8Mrrvuf28vD5+P3g0s666sU+AAAAAWJLR0QAiAUdSAAAAAlwSFlzAAAASAAAAEgARslrPgAAAAd0SU1FB+EGCQogJTCTrZ4AABUpSURBVHja7d0Je+vGeYZhYIDBQoBM7MROnCaN3dZJ27Tpvq/JpNv//0PFQmLAVxQFUQCOPvm5L/uIgsg5I+g50BCkxCQBAAAAAAAAAAAAAAAAAAAAsK3UnS+49FNPBXhUll8uuXC+EFz8sC9CSd+wo5rqvRW0rw+uLj/1HIGlsqJ0Sea964POh7dD0M77prvYHrqL1aeeJLBUH3RVeN9mXdBtWQY/BJ21vgz9piR3b/9LgN10S46qOxb7Y1fvqQs8pF3Qeeg2ndpu0zGEovnUcwQWG9bQ7lBX5zV0d3gOLitcJzQ+dOvnsvjUcwQW64L2oSqrq6B9W/WaYdNwuAZsqNyhzbslRx90miTpEPShPyinLmn6oFOChh3VIevqTYclx3l9EVza3UdMfHexOI1vASOOxffqoirKNnN1XRRtcz7LUdf9xaath02AEanLE+fSxPWPGLrpUe/0fDF1PBAOAAAAAAAAAAAAAFvKhyc85sfQlmmSlqH1n3pGwOOa4xB0fWyaukrK2h36J4kBNvkQ+qCb/mm8efh+/6Odvv7UkwIeN/5ARb/OyMNnIZn9JD5g0NRvWlcHCfrzHwx+uLovvrzri/X/RrxzxZBa8aPVgj61VZpJ0GGroH/8u7u+fHzksNUO32I3bDzlrz7faODPv1p/zDHon/zeSkGnde3Ol+dBb/Vt4af/ddfPHh95syn7zc7+bDblaqtfB7Ldr835/bWCHn9VVR7y/temTB8j6Iigo/cf9CH0v0PCJXX3VaviV46gI4KO3n/Q/dm7/gRe//Ocdfz5N4KOCDp610FfS918JxB0RNCRoaCvbbarf75Z0Jtl5zb7hYubTTn+2uqV5Zs9mGw16J9tFjQ24W5c2gJBY3P9gX7RqdwVviMQNDbXL8UXHaHjmv3rbxb6WoYgaKwpz9LxBQjOL0jQre9Pf9C/PoEfN/W/kfLWlTJ/Or+QweiFr+/zX2mCxppcfX4BgvMLEiShrP+w7zT0v4LSH9u8u3TjSt1bT9AE/e5ML0BwfkGCZOjW9SU0/aPIle+DvnWlrJovOQiaoN+F6fe1n1+QYLg4Bu3P557D7Ss5gibo92dq9fyCBPeDnl+JoAn6Hbq8AMHlBQnmQfc/yOSyIegbVyJogn6Hhhcg8G1yeUGCMehDX0LeXUyL0xD0jSsNQR8u4xA0Qb8LLlRF0R66TscXJBhaPRb9cbm7G1j1T1wLt6/UB91fcfT1zxb6WiZA0FhTtzw+v9rA9IIE4+sT9PLz01luXml+xTcgaKxp0Q9Ib/lT1ASNNRH0gwgaNxE0PhSCxodC0PhQCBofCkHjQyFofCgEjQ+FoPGhEDQ+FILGh0LQ+FAIGh8KQeNDIWh8KASt/ujuwN8+PvA396e81Z76jiFo9S1BW0bQiqBNI2hF0KYRtCJo0whaEbRpBK0I2jSCVgRtGkErgjaNoBVBm0bQiqBNI2hF0KYRtCJo0whaEbRpBK0I2jSCVgRtGkErgjaNoBVBm0bQiqBNI2hF0KYRtCJo0whaEbRpBK0I2jSCVgRtGkErgjaNoBVBm0bQiqBNI2hF0KYRtCJo0whaEbRpBK0I2rQVgs6HatMytH56c0HQEUHv4e1BN8eh2rJ2hza7vLkg6Iig9/DmoH0IfbVpcN3l+vxm+ihBRwS9hxWWHC7EP+LlEUFHBL2HtYI+jCUfCPpZBL2HtYLOxpIzDXq0/rwJGqIaUvsFSw5B0KatdYTOQ94dpovzm+ljBB0R9B7WCjqpfXfQ95c3FwQdEfQeVgu6aeuiTi9vLgg6Iug9rPfQd+rc7M0ZQUcEvQeey6EI2jSCVgRtGkErgjaNoBVBm0bQiqBNI2hF0KYRtCJo0whaEbRpBK0I2jSCVgRtGkErgjaNoBVBm0bQiqBNI2hF0KYRtCJo0whaEbRpBK0I2jSCVgRtGkErgjaNoBVBm0bQiqBNI2hF0KYRtCJo0whaEbRpBK0I2jSCVgRtGkErgjaNoBVBm0bQiqBNI2hF0KYRtCJo0whaEbRpBK0I2jSCVgRtGkErgjaNoBVBm0bQiqBNI2hF0KYRtCJo0whaEbRpBK0I2jSCVgRtGkErgjaNoBVBm0bQiqBNI2hF0KYRtCJo0whaEbRpBK0I2jSCVgRtGkErgjaNoBVBm0bQiqBNI2hF0KYRtCJo0whaEbRpBK0I2jSCVgRtGkErgjaNoBVBm0bQiqBNWy3o/BhanyZp2b2JWwk6Iug9rBZ0XTpX+KSs3aHNpq0EHRH0HlYLOrgk8VU6vKnj1q3mTdC4abWg21OSHI+uL9jFjAk6Iug9rBZ0E6q6SA8E/SyC3sN6a+jqkBU+06CrwfrzJmgIP6T2y5WCztqkL/kzDdoN1p89QUM0Q2pfrRT0cEcw7YLOu7iLaTNLjoig97DWkqMJTZKc2qT2SVLFE9EEHRH0HlZbQ59CXbRN0rR1UafTVoKOCHoP6z30nTvXh5xerZgJOiLoPfBcDkXQphG0ImjTCFoRtGkErQjaNIJWBG0aQSuCNo2gFUGbRtCKoE0jaEXQphG0ImjTCFoRtGkErQjaNIJWBG0aQSuCNo2gFUGbRtCKoE0jaEXQphG0ImjTCFoRtGkErQjaNIJWBG0aQSuCNo2gFUGbRtCKoE0jaEXQphG0ImjTCFoRtGkErQjaNIJWBG0aQSuCNo2gFUGbRtCKoE0jaEXQphG0ImjTCFoRtGkErQjaNIJWBG0aQSuCNo2gFUGbRtCKoE0jaEXQphG0ImjTCFoRtGkErQjaNIJWBG0aQSuCNo2gFUGbRtCKoE0jaEXQphG0ImjTCFoRtGkErQjaNIJWBG0aQSuCNm1Z0E3z4PAEHRH0HpYEnbWhcsVDTRN0RNB7WBB0E06uSn2bPjA8QUcEvYcFQfsycVWS1O6B4Qk6Iug9LAnaEzRBW7EgaNc2XdAZSw6CNmDRncJQtEWbPTI8QUcEvYdFp+3yzB8eOT4T9BxB72FJ0L4avHStIpRpkpah9XEjQUcEvYcFQZ9aP7h/LV8fXF0mZe0Os9UJQUcEvYdlZzkWaA/d3ccqDa5ve9pK0BFB72FB0Fm5YBwXktwNb85/jAg6Iug9LFlDV+XLSw4XjiEUzUGDXrJaeQRBQ2RDan+84Dx0qF++U+hDdxwvi4ygn0XQ21octF+45EiSPPwJS45nEfQelpzlWHKAbfp00/CnIe/+qRTTZoKOCHoPC4JOq8z17l+rOPWnopO6q7+K/wIIOiLoPSx5LkcY3b9W09ZF2wxv6vioIkFHBL2H9X4EK3UuHd/MNhJ0RNB74GcKFUGb9lLQLrhlS47bCDoi6D1whFYEbdqCoPNxUZzlDwxP0BFB7+HFoHN3KvuTdoeCH8Ei6PfvxaCzqi6GR76P/AgWQb9/S36NwUM/fDUi6Iig98CdQkXQphG0ImjTCFoRtGkErQjaNIJWBG0aQSuCNo2gFUGbRtCKoE0jaEXQphG0ImjTCFoRtGkErQjaNIJWBG0aQSuCNo2gFUGbRtCKoE0jaEXQphG0ImjTCFoRtGkErQjaNIJWBG0aQSuCNo2gFUGbRtCKoE0jaEXQphG0ImjTCFoRtGkErQjaNIJWBG0aQSuCNo2gFUGbRtCKoE0jaEXQphG0ImjTCFoRtGkErQjaNIJWBG0aQSuCNo2gFUGbRtCKoE0jaEXQphG0ImjTCFoRtGkErQjaNIJWBG0aQSuCNo2gFUGbRtCKoE0jaEXQphG0ImjTCFoRtGkErQjaNIJWBG0aQSuCNm3NoE9VkqRlaH3cRNARQe9hxaCbtgu6rN2hzaZtBB0R9B7WCzqt6ypJg0sSX08bCToi6D2sF7QvfZW4vmAXMyboiKD3sFrQhyLtgj4Q9LMIeg9rBZ22TdIFnWnQo/XnTdAQ1ZDaL1YK2tfeV4X/jCP0swh6D2sdoZ0fgv5RyJMkK6bNBB0R9B7WPA/dLTmS2nfH/ngimqAjgt7D2kE3bV3U6bSJoCOC3sPqD32nzs3eI+iIoPfAczkUQZtG0IqgTSNoRdCmEbQiaNMIWhG0aQStCNo0glYEbRpBK4I2jaAVQZtG0IqgTSNoRdCmEbQiaNMIWhG0aQStCNo0glYEbRpBK4I2jaAVQZtG0IqgTSNoRdCmEbQiaNMIWhG0aQStCNo0glYEbRpBK4I2jaAVQZtG0IqgTSNoRdCmEbQiaNMIWhG0aQStCNo0glYEbRpBK4I2jaAVQZtG0IqgTSNoRdCmEbQiaNMIWhG0aQStCNo0glYEbRpBK4I2jaAVQZtG0IqgTSNoRdCmEbQiaNMIWhG0aQStCNo0glYEbRpBK4I2jaCVwaB/9e09f/b4wH9+d+BfvWHKmyFoZTDoX2815b+4O/Cv3zDlzRC0IuiIoBVBRwS9B4JWBB0RtCLoiKD3QNCKoCOCVgQdEfQeCFoRdETQiqAjgt4DQSuCjghaEXRE0HsgaEXQ0Xc56PwY2jJN0jK0Pm4l6Iig97Ba0PWxaeoqKWt3aLNpK0FHBL2HtYJuQtodpcP3g0sSX0+bCToi6D2sFXTerzPy8FlfsIsZE3RE0HtY805hWlcHgn4WQe9hxaBPbZVmGnQ1WH/eBD1D0D0/pPbLtYJO69qdW54H7Qbrz56gZwi61wypfbXaWY6y/zMPeZJkxbSVJUdE0HtYa8lxGI/FSd3dOaziiWiCjgh6D2sF7cMgadq6qNNpM0FHBL2H1R/6Tq9WzAQdEfQeeC6HIuiIoBVBRwS9B4JWBB0RtCLoiKD3QNCKoCOCVgQdEfQeCFoRdETQiqAjgt4DQSuCjghaEXRE0HsgaEXQEUErgo4Ieg8ErQg6ImhF0BFB74GgFUFHBK0IOiLoPRC0IuiIoBVBRwS9B4JWBB0RtCLoiKD3QNCKoCOCVgQdEfQeCFoRdETQiqAjgt4DQSuCjghaEXRE0HsgaEXQEUErgo4Ieg8ErQg6ImhF0BFB74GgFUFHBK0IOiLoPRC0IuiIoBVBRwS9B4JWBB0RtCLoiKD3QNCKoCOCVgQdEfQeCFoRdETQiqAjgt4DQSuCjghaEXRE0HsgaEXQEUErgo4Ieg8ErQg6ImhF0BFBz/zlN/f81eMDE7Qi6Gi7oP/67sjfPD4wQSuCjrYL+v6UCZqgCXpE0IqgI4JWBB0R9OIpEzRBE/SIoBVBRwStCDoi6MVTJmiCJugRQSuCjghaEXRE0IunTNAETdAjglYEHRG0IuiIoBdP+TsY9M83C/pvtqrjbzcL+u+2mvLfbxV0/g8Efe2nmwX931vV8Y+bBf0/W035n7YK2v2zlaDTMrQ+vkvQEUFHdoIua3dos+ldgo4IOjITdBpckvh6ep+gI4KOzATtwuWPEUFHBB2ZCfqgQf9g8MPV/fh3d335+Mj/e3fgf3l84H+9P+U37Iz/22rK/3Z34H9/fODP/+PuyP/5yJjFkNpP1g06k6A/3yroL76864vHR/7N3YF/u9mU37AzNpvyb+8O/Js3THmDL98YdPGjVYPWJQdgWh7y7jBdfOppACupfZJU/u3jAO9C09ZFnX7qWQBrSZ371FMAAAB4pcZ10suF8+XZh2Yb9P0Fgyavv+0rBn8yZf1bXN7/mb64RruxE55OOB1Hu8ibW5/nm4edrnBv7i/ux+alT7r/C55szPPEvir0/OVCCE4+FDfo+wsGfXas7Pa+yxbs0jj4kynr3xLa/iv/8mn7Gzvh6Sebhauzpb669Xm+edjpCvfmfvtrkE/XbaqXPun+L3g6QJ3YN5wCdOMzUmUn6NnB5WcL54M+M9btfxbVkn8t88Gvp6x/y/hM2wVBP90JTz/ZYxWa2btD0E8+zzcPO13h3txvBt0cp+tWzYKgb4xRZol54w7M2v7P805wfvz/sm/1/dcM+mSsU/fHyWVF6RLfXeiu5X2/a7s3eTJsfs2Mz1N+bsbh1H/pFwZ9vROmT7Ybpx+qO/41tZ8NPgV9mcqrhr3McBr2yU6+zD1Lxx3UfdR35Q877SrocXviu+P6eUtT9DfMz7cbd3B68qc0Xjv0u7/7L/On2RjNBzhEjztweFbqZaf7MP5/2bf6/msG1dum7SE5tOlQbijrLKkK79ssKQt/bPNXBD3O+Dzl52YcnC+WB321E6ZPthunHyo5Ff1/cfAYdPrcKuzOsJcZTsM+2cmXuddtWfbLlqz1ZcjGnXa1ZBm3J7PP0/f/OML5duMOTtvK13W89rjkCGX3MT8bo7W/ij7vwHBIni45iqqTPfv+skHnt+2+FR7avHXj2mJcB/dHnWPTP6Lf3WrxkuMyuCw5ZIbBpYVfHPTVTnjyyXbH0XweUgx6/nm+edhpR13mHk79OjvN+4XJqR132uwGl+3zfVEd+nfG2513cP+xtMqna5+D7mOuZmMs+uq+b5edPjuKTB8q+/vb+bPvLxt0ftv+3vlxWBsOQY8fdoe66utIklesoS+3lqBlhv1KMTTLg54fSmWoPGTOFWW8zTxot96w0466zD2MQ2RFvzk0w2DjX9itKbo1w3n7fF+00zpr/Ku7HZyH4XvfdO1z0P0Vq9kY3v6TKsadnl/vkauvx7PvLxtUb5sN39umoH2oyuqBoM+Dv3Cn0A0/s7Yw6KudoEP54eTDbLUcg86v7tW9cdjZkmOc+zlM345Hbg36sn2+L8I86HEHJ65sQxmvfRV0HOPDBO2H1eHKQfvixm3T1vfnoy5BH/pVm6+GHypz2SuCPg/+ctBpe1wY9NVO0KGK8R5cXFzEoH2x4rBXQfdz71YN/TL80I+Snu8Pzr4lXLbP90UxBH2+3biD+7Pmrs2ma18FHccoP0DQ/bc4f7UgXeEsxzTodNvxO2nTD3I8jsu8fnf2z4RN+++ILkmL07D5NTN+6SxH/0U6hJeDfroT5CzHsMjvVrzl9VmO+VReNex5kDjskx11mXt3XO0OoV3b2fBvQ4O+bE/ma+isf6e7XVlcdvCpSLsdnE3Xvgo6jrFk/79zw4n8el7HGmc5pkGn246PFww7Nm8PybEY76undVEVZZtloeqfRNhvfs2MXzzLMd7ggZ0gZznK8YTWqTvuzc9yzKfyqmHPg8RhdUdNc+92TNHtsCRr67ptngR92Z7Mgj71/+zqurtdc9nB36vbqq3S6dpXQccxAs/jfFB8wPf8MG8+PIkwXXSf87ukq/T8SHn6zKPsT7fn53P04/bzDu6P+s+Oct56OH7qTxcf3UM/bvfoI35L7sIAb/FQ0Oljj/g5DtAAAAAAAABJ8v+rbMxwAdO6UwAAACV0RVh0ZGF0ZTpjcmVhdGUAMjAxNy0wNi0wOVQxNzozMjozNyswNzowMPSTNIQAAAAldEVYdGRhdGU6bW9kaWZ5ADIwMTctMDYtMDlUMTc6MzI6MzcrMDc6MDCFzow4AAAAIXRFWHRwczpIaVJlc0JvdW5kaW5nQm94ADUwNHg3MjArNTArNTDW4iLDAAAAE3RFWHRwczpMZXZlbABBZG9iZS0yLjAKOZN0DQAAAABJRU5ErkJggg==" />

=end html


To display as an interactive HTML table on a browser, you can add option C<--format html+datatables>.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenarios-HumanDateParsingModules>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenarios-DateTimeFormatAlami>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenarios-HumanDateParsingModules>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017, 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
