/* -*- Mode: c++ -*- */
/*
 * Copyright 2001,2002 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/*
 *  Copyright 1997 Massachusetts Institute of Technology
 *
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 *
 */

#ifndef _VRFILESOURCE_H_
#define _VRFILESOURCE_H_

#include <VrSource.h>
#include <fstream>

// This really ought to get split into a .cc file
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

// should be handled via configure
#ifdef O_LARGEFILE
#define	OUR_O_LARGEFILE	O_LARGEFILE
#else
#define	OUR_O_LARGEFILE 0
#endif

template<class oType>
class VrFileSource : public VrSource<oType> {
protected:
  FILE* fp;
  bool 	repeat;
public:
  virtual const char *name() { return "VrFileSource"; }
  virtual int work2(VrSampleRange output, void *o[]);

  VrFileSource (double sampling_freq, const char *file, bool repeat = false);
  virtual ~VrFileSource();
};

template<class oType> int
VrFileSource<oType>::work2(VrSampleRange output, void *ao[])
{
  oType *o= ((oType **)ao)[0];
  int i;
  int size = output.size;
  int index;

  sync (output.index);
  
  index = 0;
  while (size) {
    i = fread(&o[index], sizeof(oType), size, fp);
    
    size -= i;
    index += i;

    if (size == 0)		// done
      break;

    if (i > 0)			// short read, try again
      continue;

    // We got a zero from fread.  This is either EOF or error.  In
    // any event, if we're in repeat mode, seek back to the beginning
    // of the file and try again, else break

    if (!repeat)
      break;

    if (fseek(fp, 0, SEEK_SET) == -1) {
      fprintf(stderr, "[%s] fseek failed\n", __FILE__);
      exit(-1);
    }
  }

  if (size > 0){	// EOF or error
    cerr << "end of file, exiting\n";
    exit(0);
  }


  return output.size;
}

template<class oType>
VrFileSource<oType>::VrFileSource(double sampling_freq,
				  const char* filename, bool arg_repeat)
{
  setSamplingFrequency (sampling_freq);
  repeat = arg_repeat;
  int fd = -1;
  if ((fd = open (filename, O_RDONLY | OUR_O_LARGEFILE)) < 0){
    fprintf(stderr, "Could not open %s\n", filename);
    exit(1);
  }

  if((fp = fdopen (fd,"rb"))==NULL) {
    fprintf(stderr, "Could not open %s\n", filename);
    exit(1);
  }
}

template<class oType>
VrFileSource<oType>::~VrFileSource()
{
  fclose(fp);
}

#undef OUR_O_LARGEFILE
#endif
