/*
 * Copyright (C) 2016 Red Hat, Inc.
 *
 * Author: Nikos Mavrogiannopoulos
 *
 * This file is part of GnuTLS.
 *
 * The GnuTLS is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>
 *
 */

#include "gnutls_int.h"
#include <gnutls/gnutls.h>

#if defined(ENABLE_DHE) || defined(ENABLE_ANON)

#include "dh.h"

static const unsigned char ffdhe_generator = 0x02;

static const unsigned char ffdhe_params_2048[] = {
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 
	0xFF, 0xAD, 0xF8, 0x54, 0x58, 0xA2, 0xBB, 
	0x4A, 0x9A, 0xAF, 0xDC, 0x56, 0x20, 0x27, 
	0x3D, 0x3C, 0xF1, 0xD8, 0xB9, 0xC5, 0x83, 
	0xCE, 0x2D, 0x36, 0x95, 0xA9, 0xE1, 0x36, 
	0x41, 0x14, 0x64, 0x33, 0xFB, 0xCC, 0x93, 
	0x9D, 0xCE, 0x24, 0x9B, 0x3E, 0xF9, 0x7D, 
	0x2F, 0xE3, 0x63, 0x63, 0x0C, 0x75, 0xD8, 
	0xF6, 0x81, 0xB2, 0x02, 0xAE, 0xC4, 0x61, 
	0x7A, 0xD3, 0xDF, 0x1E, 0xD5, 0xD5, 0xFD, 
	0x65, 0x61, 0x24, 0x33, 0xF5, 0x1F, 0x5F, 
	0x06, 0x6E, 0xD0, 0x85, 0x63, 0x65, 0x55, 
	0x3D, 0xED, 0x1A, 0xF3, 0xB5, 0x57, 0x13, 
	0x5E, 0x7F, 0x57, 0xC9, 0x35, 0x98, 0x4F, 
	0x0C, 0x70, 0xE0, 0xE6, 0x8B, 0x77, 0xE2, 
	0xA6, 0x89, 0xDA, 0xF3, 0xEF, 0xE8, 0x72, 
	0x1D, 0xF1, 0x58, 0xA1, 0x36, 0xAD, 0xE7, 
	0x35, 0x30, 0xAC, 0xCA, 0x4F, 0x48, 0x3A, 
	0x79, 0x7A, 0xBC, 0x0A, 0xB1, 0x82, 0xB3, 
	0x24, 0xFB, 0x61, 0xD1, 0x08, 0xA9, 0x4B, 
	0xB2, 0xC8, 0xE3, 0xFB, 0xB9, 0x6A, 0xDA, 
	0xB7, 0x60, 0xD7, 0xF4, 0x68, 0x1D, 0x4F, 
	0x42, 0xA3, 0xDE, 0x39, 0x4D, 0xF4, 0xAE, 
	0x56, 0xED, 0xE7, 0x63, 0x72, 0xBB, 0x19, 
	0x0B, 0x07, 0xA7, 0xC8, 0xEE, 0x0A, 0x6D, 
	0x70, 0x9E, 0x02, 0xFC, 0xE1, 0xCD, 0xF7, 
	0xE2, 0xEC, 0xC0, 0x34, 0x04, 0xCD, 0x28, 
	0x34, 0x2F, 0x61, 0x91, 0x72, 0xFE, 0x9C, 
	0xE9, 0x85, 0x83, 0xFF, 0x8E, 0x4F, 0x12, 
	0x32, 0xEE, 0xF2, 0x81, 0x83, 0xC3, 0xFE, 
	0x3B, 0x1B, 0x4C, 0x6F, 0xAD, 0x73, 0x3B, 
	0xB5, 0xFC, 0xBC, 0x2E, 0xC2, 0x20, 0x05, 
	0xC5, 0x8E, 0xF1, 0x83, 0x7D, 0x16, 0x83, 
	0xB2, 0xC6, 0xF3, 0x4A, 0x26, 0xC1, 0xB2, 
	0xEF, 0xFA, 0x88, 0x6B, 0x42, 0x38, 0x61, 
	0x28, 0x5C, 0x97, 0xFF, 0xFF, 0xFF, 0xFF, 
	0xFF, 0xFF, 0xFF, 0xFF
};

static const unsigned char ffdhe_q_2048[] = {
	0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xD6, 0xFC, 0x2A, 0x2C, 0x51, 0x5D,
	0xA5, 0x4D, 0x57, 0xEE, 0x2B, 0x10, 0x13,
	0x9E, 0x9E, 0x78, 0xEC, 0x5C, 0xE2, 0xC1,
	0xE7, 0x16, 0x9B, 0x4A, 0xD4, 0xF0, 0x9B,
	0x20, 0x8A, 0x32, 0x19, 0xFD, 0xE6, 0x49,
	0xCE, 0xE7, 0x12, 0x4D, 0x9F, 0x7C, 0xBE,
	0x97, 0xF1, 0xB1, 0xB1, 0x86, 0x3A, 0xEC,
	0x7B, 0x40, 0xD9, 0x01, 0x57, 0x62, 0x30,
	0xBD, 0x69, 0xEF, 0x8F, 0x6A, 0xEA, 0xFE,
	0xB2, 0xB0, 0x92, 0x19, 0xFA, 0x8F, 0xAF,
	0x83, 0x37, 0x68, 0x42, 0xB1, 0xB2, 0xAA,
	0x9E, 0xF6, 0x8D, 0x79, 0xDA, 0xAB, 0x89,
	0xAF, 0x3F, 0xAB, 0xE4, 0x9A, 0xCC, 0x27,
	0x86, 0x38, 0x70, 0x73, 0x45, 0xBB, 0xF1,
	0x53, 0x44, 0xED, 0x79, 0xF7, 0xF4, 0x39,
	0x0E, 0xF8, 0xAC, 0x50, 0x9B, 0x56, 0xF3,
	0x9A, 0x98, 0x56, 0x65, 0x27, 0xA4, 0x1D,
	0x3C, 0xBD, 0x5E, 0x05, 0x58, 0xC1, 0x59,
	0x92, 0x7D, 0xB0, 0xE8, 0x84, 0x54, 0xA5,
	0xD9, 0x64, 0x71, 0xFD, 0xDC, 0xB5, 0x6D,
	0x5B, 0xB0, 0x6B, 0xFA, 0x34, 0x0E, 0xA7,
	0xA1, 0x51, 0xEF, 0x1C, 0xA6, 0xFA, 0x57,
	0x2B, 0x76, 0xF3, 0xB1, 0xB9, 0x5D, 0x8C,
	0x85, 0x83, 0xD3, 0xE4, 0x77, 0x05, 0x36,
	0xB8, 0x4F, 0x01, 0x7E, 0x70, 0xE6, 0xFB,
	0xF1, 0x76, 0x60, 0x1A, 0x02, 0x66, 0x94,
	0x1A, 0x17, 0xB0, 0xC8, 0xB9, 0x7F, 0x4E,
	0x74, 0xC2, 0xC1, 0xFF, 0xC7, 0x27, 0x89,
	0x19, 0x77, 0x79, 0x40, 0xC1, 0xE1, 0xFF,
	0x1D, 0x8D, 0xA6, 0x37, 0xD6, 0xB9, 0x9D,
	0xDA, 0xFE, 0x5E, 0x17, 0x61, 0x10, 0x02,
	0xE2, 0xC7, 0x78, 0xC1, 0xBE, 0x8B, 0x41,
	0xD9, 0x63, 0x79, 0xA5, 0x13, 0x60, 0xD9,
	0x77, 0xFD, 0x44, 0x35, 0xA1, 0x1C, 0x30,
	0x94, 0x2E, 0x4B, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF
};

const gnutls_datum_t gnutls_ffdhe_2048_group_prime = {
	(void *) ffdhe_params_2048, sizeof(ffdhe_params_2048)
};
const gnutls_datum_t gnutls_ffdhe_2048_group_q = {
	(void *) ffdhe_q_2048, sizeof(ffdhe_q_2048)
};
const gnutls_datum_t gnutls_ffdhe_2048_group_generator = {
	(void *) &ffdhe_generator, sizeof(ffdhe_generator)
};
const unsigned int gnutls_ffdhe_2048_key_bits = 256;

static const unsigned char ffdhe_params_3072[] = {
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 
	0xFF, 0xAD, 0xF8, 0x54, 0x58, 0xA2, 0xBB, 
	0x4A, 0x9A, 0xAF, 0xDC, 0x56, 0x20, 0x27, 
	0x3D, 0x3C, 0xF1, 0xD8, 0xB9, 0xC5, 0x83, 
	0xCE, 0x2D, 0x36, 0x95, 0xA9, 0xE1, 0x36, 
	0x41, 0x14, 0x64, 0x33, 0xFB, 0xCC, 0x93, 
	0x9D, 0xCE, 0x24, 0x9B, 0x3E, 0xF9, 0x7D, 
	0x2F, 0xE3, 0x63, 0x63, 0x0C, 0x75, 0xD8, 
	0xF6, 0x81, 0xB2, 0x02, 0xAE, 0xC4, 0x61, 
	0x7A, 0xD3, 0xDF, 0x1E, 0xD5, 0xD5, 0xFD, 
	0x65, 0x61, 0x24, 0x33, 0xF5, 0x1F, 0x5F, 
	0x06, 0x6E, 0xD0, 0x85, 0x63, 0x65, 0x55, 
	0x3D, 0xED, 0x1A, 0xF3, 0xB5, 0x57, 0x13, 
	0x5E, 0x7F, 0x57, 0xC9, 0x35, 0x98, 0x4F, 
	0x0C, 0x70, 0xE0, 0xE6, 0x8B, 0x77, 0xE2, 
	0xA6, 0x89, 0xDA, 0xF3, 0xEF, 0xE8, 0x72, 
	0x1D, 0xF1, 0x58, 0xA1, 0x36, 0xAD, 0xE7, 
	0x35, 0x30, 0xAC, 0xCA, 0x4F, 0x48, 0x3A, 
	0x79, 0x7A, 0xBC, 0x0A, 0xB1, 0x82, 0xB3, 
	0x24, 0xFB, 0x61, 0xD1, 0x08, 0xA9, 0x4B, 
	0xB2, 0xC8, 0xE3, 0xFB, 0xB9, 0x6A, 0xDA, 
	0xB7, 0x60, 0xD7, 0xF4, 0x68, 0x1D, 0x4F, 
	0x42, 0xA3, 0xDE, 0x39, 0x4D, 0xF4, 0xAE, 
	0x56, 0xED, 0xE7, 0x63, 0x72, 0xBB, 0x19, 
	0x0B, 0x07, 0xA7, 0xC8, 0xEE, 0x0A, 0x6D, 
	0x70, 0x9E, 0x02, 0xFC, 0xE1, 0xCD, 0xF7, 
	0xE2, 0xEC, 0xC0, 0x34, 0x04, 0xCD, 0x28, 
	0x34, 0x2F, 0x61, 0x91, 0x72, 0xFE, 0x9C, 
	0xE9, 0x85, 0x83, 0xFF, 0x8E, 0x4F, 0x12, 
	0x32, 0xEE, 0xF2, 0x81, 0x83, 0xC3, 0xFE, 
	0x3B, 0x1B, 0x4C, 0x6F, 0xAD, 0x73, 0x3B, 
	0xB5, 0xFC, 0xBC, 0x2E, 0xC2, 0x20, 0x05, 
	0xC5, 0x8E, 0xF1, 0x83, 0x7D, 0x16, 0x83, 
	0xB2, 0xC6, 0xF3, 0x4A, 0x26, 0xC1, 0xB2, 
	0xEF, 0xFA, 0x88, 0x6B, 0x42, 0x38, 0x61, 
	0x1F, 0xCF, 0xDC, 0xDE, 0x35, 0x5B, 0x3B, 
	0x65, 0x19, 0x03, 0x5B, 0xBC, 0x34, 0xF4, 
	0xDE, 0xF9, 0x9C, 0x02, 0x38, 0x61, 0xB4, 
	0x6F, 0xC9, 0xD6, 0xE6, 0xC9, 0x07, 0x7A, 
	0xD9, 0x1D, 0x26, 0x91, 0xF7, 0xF7, 0xEE, 
	0x59, 0x8C, 0xB0, 0xFA, 0xC1, 0x86, 0xD9, 
	0x1C, 0xAE, 0xFE, 0x13, 0x09, 0x85, 0x13, 
	0x92, 0x70, 0xB4, 0x13, 0x0C, 0x93, 0xBC, 
	0x43, 0x79, 0x44, 0xF4, 0xFD, 0x44, 0x52, 
	0xE2, 0xD7, 0x4D, 0xD3, 0x64, 0xF2, 0xE2, 
	0x1E, 0x71, 0xF5, 0x4B, 0xFF, 0x5C, 0xAE, 
	0x82, 0xAB, 0x9C, 0x9D, 0xF6, 0x9E, 0xE8, 
	0x6D, 0x2B, 0xC5, 0x22, 0x36, 0x3A, 0x0D, 
	0xAB, 0xC5, 0x21, 0x97, 0x9B, 0x0D, 0xEA, 
	0xDA, 0x1D, 0xBF, 0x9A, 0x42, 0xD5, 0xC4, 
	0x48, 0x4E, 0x0A, 0xBC, 0xD0, 0x6B, 0xFA, 
	0x53, 0xDD, 0xEF, 0x3C, 0x1B, 0x20, 0xEE, 
	0x3F, 0xD5, 0x9D, 0x7C, 0x25, 0xE4, 0x1D, 
	0x2B, 0x66, 0xC6, 0x2E, 0x37, 0xFF, 0xFF, 
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

static const unsigned char ffdhe_q_3072[] = {
	0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xD6, 0xFC, 0x2A, 0x2C, 0x51, 0x5D,
	0xA5, 0x4D, 0x57, 0xEE, 0x2B, 0x10, 0x13,
	0x9E, 0x9E, 0x78, 0xEC, 0x5C, 0xE2, 0xC1,
	0xE7, 0x16, 0x9B, 0x4A, 0xD4, 0xF0, 0x9B,
	0x20, 0x8A, 0x32, 0x19, 0xFD, 0xE6, 0x49,
	0xCE, 0xE7, 0x12, 0x4D, 0x9F, 0x7C, 0xBE,
	0x97, 0xF1, 0xB1, 0xB1, 0x86, 0x3A, 0xEC,
	0x7B, 0x40, 0xD9, 0x01, 0x57, 0x62, 0x30,
	0xBD, 0x69, 0xEF, 0x8F, 0x6A, 0xEA, 0xFE,
	0xB2, 0xB0, 0x92, 0x19, 0xFA, 0x8F, 0xAF,
	0x83, 0x37, 0x68, 0x42, 0xB1, 0xB2, 0xAA,
	0x9E, 0xF6, 0x8D, 0x79, 0xDA, 0xAB, 0x89,
	0xAF, 0x3F, 0xAB, 0xE4, 0x9A, 0xCC, 0x27,
	0x86, 0x38, 0x70, 0x73, 0x45, 0xBB, 0xF1,
	0x53, 0x44, 0xED, 0x79, 0xF7, 0xF4, 0x39,
	0x0E, 0xF8, 0xAC, 0x50, 0x9B, 0x56, 0xF3,
	0x9A, 0x98, 0x56, 0x65, 0x27, 0xA4, 0x1D,
	0x3C, 0xBD, 0x5E, 0x05, 0x58, 0xC1, 0x59,
	0x92, 0x7D, 0xB0, 0xE8, 0x84, 0x54, 0xA5,
	0xD9, 0x64, 0x71, 0xFD, 0xDC, 0xB5, 0x6D,
	0x5B, 0xB0, 0x6B, 0xFA, 0x34, 0x0E, 0xA7,
	0xA1, 0x51, 0xEF, 0x1C, 0xA6, 0xFA, 0x57,
	0x2B, 0x76, 0xF3, 0xB1, 0xB9, 0x5D, 0x8C,
	0x85, 0x83, 0xD3, 0xE4, 0x77, 0x05, 0x36,
	0xB8, 0x4F, 0x01, 0x7E, 0x70, 0xE6, 0xFB,
	0xF1, 0x76, 0x60, 0x1A, 0x02, 0x66, 0x94,
	0x1A, 0x17, 0xB0, 0xC8, 0xB9, 0x7F, 0x4E,
	0x74, 0xC2, 0xC1, 0xFF, 0xC7, 0x27, 0x89,
	0x19, 0x77, 0x79, 0x40, 0xC1, 0xE1, 0xFF,
	0x1D, 0x8D, 0xA6, 0x37, 0xD6, 0xB9, 0x9D,
	0xDA, 0xFE, 0x5E, 0x17, 0x61, 0x10, 0x02,
	0xE2, 0xC7, 0x78, 0xC1, 0xBE, 0x8B, 0x41,
	0xD9, 0x63, 0x79, 0xA5, 0x13, 0x60, 0xD9,
	0x77, 0xFD, 0x44, 0x35, 0xA1, 0x1C, 0x30,
	0x8F, 0xE7, 0xEE, 0x6F, 0x1A, 0xAD, 0x9D,
	0xB2, 0x8C, 0x81, 0xAD, 0xDE, 0x1A, 0x7A,
	0x6F, 0x7C, 0xCE, 0x01, 0x1C, 0x30, 0xDA,
	0x37, 0xE4, 0xEB, 0x73, 0x64, 0x83, 0xBD,
	0x6C, 0x8E, 0x93, 0x48, 0xFB, 0xFB, 0xF7,
	0x2C, 0xC6, 0x58, 0x7D, 0x60, 0xC3, 0x6C,
	0x8E, 0x57, 0x7F, 0x09, 0x84, 0xC2, 0x89,
	0xC9, 0x38, 0x5A, 0x09, 0x86, 0x49, 0xDE,
	0x21, 0xBC, 0xA2, 0x7A, 0x7E, 0xA2, 0x29,
	0x71, 0x6B, 0xA6, 0xE9, 0xB2, 0x79, 0x71,
	0x0F, 0x38, 0xFA, 0xA5, 0xFF, 0xAE, 0x57,
	0x41, 0x55, 0xCE, 0x4E, 0xFB, 0x4F, 0x74,
	0x36, 0x95, 0xE2, 0x91, 0x1B, 0x1D, 0x06,
	0xD5, 0xE2, 0x90, 0xCB, 0xCD, 0x86, 0xF5,
	0x6D, 0x0E, 0xDF, 0xCD, 0x21, 0x6A, 0xE2,
	0x24, 0x27, 0x05, 0x5E, 0x68, 0x35, 0xFD,
	0x29, 0xEE, 0xF7, 0x9E, 0x0D, 0x90, 0x77,
	0x1F, 0xEA, 0xCE, 0xBE, 0x12, 0xF2, 0x0E,
	0x95, 0xB3, 0x63, 0x17, 0x1B, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

const gnutls_datum_t gnutls_ffdhe_3072_group_prime = {
	(void *) ffdhe_params_3072, sizeof(ffdhe_params_3072)
};
const gnutls_datum_t gnutls_ffdhe_3072_group_q = {
	(void *) ffdhe_q_3072, sizeof(ffdhe_q_3072)
};
const gnutls_datum_t gnutls_ffdhe_3072_group_generator = {
	(void *) &ffdhe_generator, sizeof(ffdhe_generator)
};
const unsigned int gnutls_ffdhe_3072_key_bits = 276;

static const unsigned char ffdhe_params_4096[] = {
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 
	0xFF, 0xAD, 0xF8, 0x54, 0x58, 0xA2, 0xBB, 
	0x4A, 0x9A, 0xAF, 0xDC, 0x56, 0x20, 0x27, 
	0x3D, 0x3C, 0xF1, 0xD8, 0xB9, 0xC5, 0x83, 
	0xCE, 0x2D, 0x36, 0x95, 0xA9, 0xE1, 0x36, 
	0x41, 0x14, 0x64, 0x33, 0xFB, 0xCC, 0x93, 
	0x9D, 0xCE, 0x24, 0x9B, 0x3E, 0xF9, 0x7D, 
	0x2F, 0xE3, 0x63, 0x63, 0x0C, 0x75, 0xD8, 
	0xF6, 0x81, 0xB2, 0x02, 0xAE, 0xC4, 0x61, 
	0x7A, 0xD3, 0xDF, 0x1E, 0xD5, 0xD5, 0xFD, 
	0x65, 0x61, 0x24, 0x33, 0xF5, 0x1F, 0x5F, 
	0x06, 0x6E, 0xD0, 0x85, 0x63, 0x65, 0x55, 
	0x3D, 0xED, 0x1A, 0xF3, 0xB5, 0x57, 0x13, 
	0x5E, 0x7F, 0x57, 0xC9, 0x35, 0x98, 0x4F, 
	0x0C, 0x70, 0xE0, 0xE6, 0x8B, 0x77, 0xE2, 
	0xA6, 0x89, 0xDA, 0xF3, 0xEF, 0xE8, 0x72, 
	0x1D, 0xF1, 0x58, 0xA1, 0x36, 0xAD, 0xE7, 
	0x35, 0x30, 0xAC, 0xCA, 0x4F, 0x48, 0x3A, 
	0x79, 0x7A, 0xBC, 0x0A, 0xB1, 0x82, 0xB3, 
	0x24, 0xFB, 0x61, 0xD1, 0x08, 0xA9, 0x4B, 
	0xB2, 0xC8, 0xE3, 0xFB, 0xB9, 0x6A, 0xDA, 
	0xB7, 0x60, 0xD7, 0xF4, 0x68, 0x1D, 0x4F, 
	0x42, 0xA3, 0xDE, 0x39, 0x4D, 0xF4, 0xAE, 
	0x56, 0xED, 0xE7, 0x63, 0x72, 0xBB, 0x19, 
	0x0B, 0x07, 0xA7, 0xC8, 0xEE, 0x0A, 0x6D, 
	0x70, 0x9E, 0x02, 0xFC, 0xE1, 0xCD, 0xF7, 
	0xE2, 0xEC, 0xC0, 0x34, 0x04, 0xCD, 0x28, 
	0x34, 0x2F, 0x61, 0x91, 0x72, 0xFE, 0x9C, 
	0xE9, 0x85, 0x83, 0xFF, 0x8E, 0x4F, 0x12, 
	0x32, 0xEE, 0xF2, 0x81, 0x83, 0xC3, 0xFE, 
	0x3B, 0x1B, 0x4C, 0x6F, 0xAD, 0x73, 0x3B, 
	0xB5, 0xFC, 0xBC, 0x2E, 0xC2, 0x20, 0x05, 
	0xC5, 0x8E, 0xF1, 0x83, 0x7D, 0x16, 0x83, 
	0xB2, 0xC6, 0xF3, 0x4A, 0x26, 0xC1, 0xB2, 
	0xEF, 0xFA, 0x88, 0x6B, 0x42, 0x38, 0x61, 
	0x1F, 0xCF, 0xDC, 0xDE, 0x35, 0x5B, 0x3B, 
	0x65, 0x19, 0x03, 0x5B, 0xBC, 0x34, 0xF4, 
	0xDE, 0xF9, 0x9C, 0x02, 0x38, 0x61, 0xB4, 
	0x6F, 0xC9, 0xD6, 0xE6, 0xC9, 0x07, 0x7A, 
	0xD9, 0x1D, 0x26, 0x91, 0xF7, 0xF7, 0xEE, 
	0x59, 0x8C, 0xB0, 0xFA, 0xC1, 0x86, 0xD9, 
	0x1C, 0xAE, 0xFE, 0x13, 0x09, 0x85, 0x13, 
	0x92, 0x70, 0xB4, 0x13, 0x0C, 0x93, 0xBC, 
	0x43, 0x79, 0x44, 0xF4, 0xFD, 0x44, 0x52, 
	0xE2, 0xD7, 0x4D, 0xD3, 0x64, 0xF2, 0xE2, 
	0x1E, 0x71, 0xF5, 0x4B, 0xFF, 0x5C, 0xAE, 
	0x82, 0xAB, 0x9C, 0x9D, 0xF6, 0x9E, 0xE8, 
	0x6D, 0x2B, 0xC5, 0x22, 0x36, 0x3A, 0x0D, 
	0xAB, 0xC5, 0x21, 0x97, 0x9B, 0x0D, 0xEA, 
	0xDA, 0x1D, 0xBF, 0x9A, 0x42, 0xD5, 0xC4, 
	0x48, 0x4E, 0x0A, 0xBC, 0xD0, 0x6B, 0xFA, 
	0x53, 0xDD, 0xEF, 0x3C, 0x1B, 0x20, 0xEE, 
	0x3F, 0xD5, 0x9D, 0x7C, 0x25, 0xE4, 0x1D, 
	0x2B, 0x66, 0x9E, 0x1E, 0xF1, 0x6E, 0x6F, 
	0x52, 0xC3, 0x16, 0x4D, 0xF4, 0xFB, 0x79, 
	0x30, 0xE9, 0xE4, 0xE5, 0x88, 0x57, 0xB6, 
	0xAC, 0x7D, 0x5F, 0x42, 0xD6, 0x9F, 0x6D, 
	0x18, 0x77, 0x63, 0xCF, 0x1D, 0x55, 0x03, 
	0x40, 0x04, 0x87, 0xF5, 0x5B, 0xA5, 0x7E, 
	0x31, 0xCC, 0x7A, 0x71, 0x35, 0xC8, 0x86, 
	0xEF, 0xB4, 0x31, 0x8A, 0xED, 0x6A, 0x1E, 
	0x01, 0x2D, 0x9E, 0x68, 0x32, 0xA9, 0x07, 
	0x60, 0x0A, 0x91, 0x81, 0x30, 0xC4, 0x6D, 
	0xC7, 0x78, 0xF9, 0x71, 0xAD, 0x00, 0x38, 
	0x09, 0x29, 0x99, 0xA3, 0x33, 0xCB, 0x8B, 
	0x7A, 0x1A, 0x1D, 0xB9, 0x3D, 0x71, 0x40, 
	0x00, 0x3C, 0x2A, 0x4E, 0xCE, 0xA9, 0xF9, 
	0x8D, 0x0A, 0xCC, 0x0A, 0x82, 0x91, 0xCD, 
	0xCE, 0xC9, 0x7D, 0xCF, 0x8E, 0xC9, 0xB5, 
	0x5A, 0x7F, 0x88, 0xA4, 0x6B, 0x4D, 0xB5, 
	0xA8, 0x51, 0xF4, 0x41, 0x82, 0xE1, 0xC6, 
	0x8A, 0x00, 0x7E, 0x5E, 0x65, 0x5F, 0x6A, 
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 
	0xFF
};

static const unsigned char ffdhe_q_4096[] = {
	0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xD6, 0xFC, 0x2A, 0x2C, 0x51, 0x5D,
	0xA5, 0x4D, 0x57, 0xEE, 0x2B, 0x10, 0x13,
	0x9E, 0x9E, 0x78, 0xEC, 0x5C, 0xE2, 0xC1,
	0xE7, 0x16, 0x9B, 0x4A, 0xD4, 0xF0, 0x9B,
	0x20, 0x8A, 0x32, 0x19, 0xFD, 0xE6, 0x49,
	0xCE, 0xE7, 0x12, 0x4D, 0x9F, 0x7C, 0xBE,
	0x97, 0xF1, 0xB1, 0xB1, 0x86, 0x3A, 0xEC,
	0x7B, 0x40, 0xD9, 0x01, 0x57, 0x62, 0x30,
	0xBD, 0x69, 0xEF, 0x8F, 0x6A, 0xEA, 0xFE,
	0xB2, 0xB0, 0x92, 0x19, 0xFA, 0x8F, 0xAF,
	0x83, 0x37, 0x68, 0x42, 0xB1, 0xB2, 0xAA,
	0x9E, 0xF6, 0x8D, 0x79, 0xDA, 0xAB, 0x89,
	0xAF, 0x3F, 0xAB, 0xE4, 0x9A, 0xCC, 0x27,
	0x86, 0x38, 0x70, 0x73, 0x45, 0xBB, 0xF1,
	0x53, 0x44, 0xED, 0x79, 0xF7, 0xF4, 0x39,
	0x0E, 0xF8, 0xAC, 0x50, 0x9B, 0x56, 0xF3,
	0x9A, 0x98, 0x56, 0x65, 0x27, 0xA4, 0x1D,
	0x3C, 0xBD, 0x5E, 0x05, 0x58, 0xC1, 0x59,
	0x92, 0x7D, 0xB0, 0xE8, 0x84, 0x54, 0xA5,
	0xD9, 0x64, 0x71, 0xFD, 0xDC, 0xB5, 0x6D,
	0x5B, 0xB0, 0x6B, 0xFA, 0x34, 0x0E, 0xA7,
	0xA1, 0x51, 0xEF, 0x1C, 0xA6, 0xFA, 0x57,
	0x2B, 0x76, 0xF3, 0xB1, 0xB9, 0x5D, 0x8C,
	0x85, 0x83, 0xD3, 0xE4, 0x77, 0x05, 0x36,
	0xB8, 0x4F, 0x01, 0x7E, 0x70, 0xE6, 0xFB,
	0xF1, 0x76, 0x60, 0x1A, 0x02, 0x66, 0x94,
	0x1A, 0x17, 0xB0, 0xC8, 0xB9, 0x7F, 0x4E,
	0x74, 0xC2, 0xC1, 0xFF, 0xC7, 0x27, 0x89,
	0x19, 0x77, 0x79, 0x40, 0xC1, 0xE1, 0xFF,
	0x1D, 0x8D, 0xA6, 0x37, 0xD6, 0xB9, 0x9D,
	0xDA, 0xFE, 0x5E, 0x17, 0x61, 0x10, 0x02,
	0xE2, 0xC7, 0x78, 0xC1, 0xBE, 0x8B, 0x41,
	0xD9, 0x63, 0x79, 0xA5, 0x13, 0x60, 0xD9,
	0x77, 0xFD, 0x44, 0x35, 0xA1, 0x1C, 0x30,
	0x8F, 0xE7, 0xEE, 0x6F, 0x1A, 0xAD, 0x9D,
	0xB2, 0x8C, 0x81, 0xAD, 0xDE, 0x1A, 0x7A,
	0x6F, 0x7C, 0xCE, 0x01, 0x1C, 0x30, 0xDA,
	0x37, 0xE4, 0xEB, 0x73, 0x64, 0x83, 0xBD,
	0x6C, 0x8E, 0x93, 0x48, 0xFB, 0xFB, 0xF7,
	0x2C, 0xC6, 0x58, 0x7D, 0x60, 0xC3, 0x6C,
	0x8E, 0x57, 0x7F, 0x09, 0x84, 0xC2, 0x89,
	0xC9, 0x38, 0x5A, 0x09, 0x86, 0x49, 0xDE,
	0x21, 0xBC, 0xA2, 0x7A, 0x7E, 0xA2, 0x29,
	0x71, 0x6B, 0xA6, 0xE9, 0xB2, 0x79, 0x71,
	0x0F, 0x38, 0xFA, 0xA5, 0xFF, 0xAE, 0x57,
	0x41, 0x55, 0xCE, 0x4E, 0xFB, 0x4F, 0x74,
	0x36, 0x95, 0xE2, 0x91, 0x1B, 0x1D, 0x06,
	0xD5, 0xE2, 0x90, 0xCB, 0xCD, 0x86, 0xF5,
	0x6D, 0x0E, 0xDF, 0xCD, 0x21, 0x6A, 0xE2,
	0x24, 0x27, 0x05, 0x5E, 0x68, 0x35, 0xFD,
	0x29, 0xEE, 0xF7, 0x9E, 0x0D, 0x90, 0x77,
	0x1F, 0xEA, 0xCE, 0xBE, 0x12, 0xF2, 0x0E,
	0x95, 0xB3, 0x4F, 0x0F, 0x78, 0xB7, 0x37,
	0xA9, 0x61, 0x8B, 0x26, 0xFA, 0x7D, 0xBC,
	0x98, 0x74, 0xF2, 0x72, 0xC4, 0x2B, 0xDB,
	0x56, 0x3E, 0xAF, 0xA1, 0x6B, 0x4F, 0xB6,
	0x8C, 0x3B, 0xB1, 0xE7, 0x8E, 0xAA, 0x81,
	0xA0, 0x02, 0x43, 0xFA, 0xAD, 0xD2, 0xBF,
	0x18, 0xE6, 0x3D, 0x38, 0x9A, 0xE4, 0x43,
	0x77, 0xDA, 0x18, 0xC5, 0x76, 0xB5, 0x0F,
	0x00, 0x96, 0xCF, 0x34, 0x19, 0x54, 0x83,
	0xB0, 0x05, 0x48, 0xC0, 0x98, 0x62, 0x36,
	0xE3, 0xBC, 0x7C, 0xB8, 0xD6, 0x80, 0x1C,
	0x04, 0x94, 0xCC, 0xD1, 0x99, 0xE5, 0xC5,
	0xBD, 0x0D, 0x0E, 0xDC, 0x9E, 0xB8, 0xA0,
	0x00, 0x1E, 0x15, 0x27, 0x67, 0x54, 0xFC,
	0xC6, 0x85, 0x66, 0x05, 0x41, 0x48, 0xE6,
	0xE7, 0x64, 0xBE, 0xE7, 0xC7, 0x64, 0xDA,
	0xAD, 0x3F, 0xC4, 0x52, 0x35, 0xA6, 0xDA,
	0xD4, 0x28, 0xFA, 0x20, 0xC1, 0x70, 0xE3,
	0x45, 0x00, 0x3F, 0x2F, 0x32, 0xAF, 0xB5,
	0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF
};

const gnutls_datum_t gnutls_ffdhe_4096_group_prime = {
	(void *) ffdhe_params_4096, sizeof(ffdhe_params_4096)
};
const gnutls_datum_t gnutls_ffdhe_4096_group_q = {
	(void *) ffdhe_q_4096, sizeof(ffdhe_q_4096)
};
const gnutls_datum_t gnutls_ffdhe_4096_group_generator = {
	(void *) &ffdhe_generator, sizeof(ffdhe_generator)
};
const unsigned int gnutls_ffdhe_4096_key_bits = 336;

static const unsigned char ffdhe_params_6144[] = {
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xAD, 0xF8, 0x54, 0x58, 0xA2, 0xBB,
	0x4A, 0x9A, 0xAF, 0xDC, 0x56, 0x20, 0x27,
	0x3D, 0x3C, 0xF1, 0xD8, 0xB9, 0xC5, 0x83,
	0xCE, 0x2D, 0x36, 0x95, 0xA9, 0xE1, 0x36,
	0x41, 0x14, 0x64, 0x33, 0xFB, 0xCC, 0x93,
	0x9D, 0xCE, 0x24, 0x9B, 0x3E, 0xF9, 0x7D,
	0x2F, 0xE3, 0x63, 0x63, 0x0C, 0x75, 0xD8,
	0xF6, 0x81, 0xB2, 0x02, 0xAE, 0xC4, 0x61,
	0x7A, 0xD3, 0xDF, 0x1E, 0xD5, 0xD5, 0xFD,
	0x65, 0x61, 0x24, 0x33, 0xF5, 0x1F, 0x5F,
	0x06, 0x6E, 0xD0, 0x85, 0x63, 0x65, 0x55,
	0x3D, 0xED, 0x1A, 0xF3, 0xB5, 0x57, 0x13,
	0x5E, 0x7F, 0x57, 0xC9, 0x35, 0x98, 0x4F,
	0x0C, 0x70, 0xE0, 0xE6, 0x8B, 0x77, 0xE2,
	0xA6, 0x89, 0xDA, 0xF3, 0xEF, 0xE8, 0x72,
	0x1D, 0xF1, 0x58, 0xA1, 0x36, 0xAD, 0xE7,
	0x35, 0x30, 0xAC, 0xCA, 0x4F, 0x48, 0x3A,
	0x79, 0x7A, 0xBC, 0x0A, 0xB1, 0x82, 0xB3,
	0x24, 0xFB, 0x61, 0xD1, 0x08, 0xA9, 0x4B,
	0xB2, 0xC8, 0xE3, 0xFB, 0xB9, 0x6A, 0xDA,
	0xB7, 0x60, 0xD7, 0xF4, 0x68, 0x1D, 0x4F,
	0x42, 0xA3, 0xDE, 0x39, 0x4D, 0xF4, 0xAE,
	0x56, 0xED, 0xE7, 0x63, 0x72, 0xBB, 0x19,
	0x0B, 0x07, 0xA7, 0xC8, 0xEE, 0x0A, 0x6D,
	0x70, 0x9E, 0x02, 0xFC, 0xE1, 0xCD, 0xF7,
	0xE2, 0xEC, 0xC0, 0x34, 0x04, 0xCD, 0x28,
	0x34, 0x2F, 0x61, 0x91, 0x72, 0xFE, 0x9C,
	0xE9, 0x85, 0x83, 0xFF, 0x8E, 0x4F, 0x12,
	0x32, 0xEE, 0xF2, 0x81, 0x83, 0xC3, 0xFE,
	0x3B, 0x1B, 0x4C, 0x6F, 0xAD, 0x73, 0x3B,
	0xB5, 0xFC, 0xBC, 0x2E, 0xC2, 0x20, 0x05,
	0xC5, 0x8E, 0xF1, 0x83, 0x7D, 0x16, 0x83,
	0xB2, 0xC6, 0xF3, 0x4A, 0x26, 0xC1, 0xB2,
	0xEF, 0xFA, 0x88, 0x6B, 0x42, 0x38, 0x61,
	0x1F, 0xCF, 0xDC, 0xDE, 0x35, 0x5B, 0x3B,
	0x65, 0x19, 0x03, 0x5B, 0xBC, 0x34, 0xF4,
	0xDE, 0xF9, 0x9C, 0x02, 0x38, 0x61, 0xB4,
	0x6F, 0xC9, 0xD6, 0xE6, 0xC9, 0x07, 0x7A,
	0xD9, 0x1D, 0x26, 0x91, 0xF7, 0xF7, 0xEE,
	0x59, 0x8C, 0xB0, 0xFA, 0xC1, 0x86, 0xD9,
	0x1C, 0xAE, 0xFE, 0x13, 0x09, 0x85, 0x13,
	0x92, 0x70, 0xB4, 0x13, 0x0C, 0x93, 0xBC,
	0x43, 0x79, 0x44, 0xF4, 0xFD, 0x44, 0x52,
	0xE2, 0xD7, 0x4D, 0xD3, 0x64, 0xF2, 0xE2,
	0x1E, 0x71, 0xF5, 0x4B, 0xFF, 0x5C, 0xAE,
	0x82, 0xAB, 0x9C, 0x9D, 0xF6, 0x9E, 0xE8,
	0x6D, 0x2B, 0xC5, 0x22, 0x36, 0x3A, 0x0D,
	0xAB, 0xC5, 0x21, 0x97, 0x9B, 0x0D, 0xEA,
	0xDA, 0x1D, 0xBF, 0x9A, 0x42, 0xD5, 0xC4,
	0x48, 0x4E, 0x0A, 0xBC, 0xD0, 0x6B, 0xFA,
	0x53, 0xDD, 0xEF, 0x3C, 0x1B, 0x20, 0xEE,
	0x3F, 0xD5, 0x9D, 0x7C, 0x25, 0xE4, 0x1D,
	0x2B, 0x66, 0x9E, 0x1E, 0xF1, 0x6E, 0x6F,
	0x52, 0xC3, 0x16, 0x4D, 0xF4, 0xFB, 0x79,
	0x30, 0xE9, 0xE4, 0xE5, 0x88, 0x57, 0xB6,
	0xAC, 0x7D, 0x5F, 0x42, 0xD6, 0x9F, 0x6D,
	0x18, 0x77, 0x63, 0xCF, 0x1D, 0x55, 0x03,
	0x40, 0x04, 0x87, 0xF5, 0x5B, 0xA5, 0x7E,
	0x31, 0xCC, 0x7A, 0x71, 0x35, 0xC8, 0x86,
	0xEF, 0xB4, 0x31, 0x8A, 0xED, 0x6A, 0x1E,
	0x01, 0x2D, 0x9E, 0x68, 0x32, 0xA9, 0x07,
	0x60, 0x0A, 0x91, 0x81, 0x30, 0xC4, 0x6D,
	0xC7, 0x78, 0xF9, 0x71, 0xAD, 0x00, 0x38,
	0x09, 0x29, 0x99, 0xA3, 0x33, 0xCB, 0x8B,
	0x7A, 0x1A, 0x1D, 0xB9, 0x3D, 0x71, 0x40,
	0x00, 0x3C, 0x2A, 0x4E, 0xCE, 0xA9, 0xF9,
	0x8D, 0x0A, 0xCC, 0x0A, 0x82, 0x91, 0xCD,
	0xCE, 0xC9, 0x7D, 0xCF, 0x8E, 0xC9, 0xB5,
	0x5A, 0x7F, 0x88, 0xA4, 0x6B, 0x4D, 0xB5,
	0xA8, 0x51, 0xF4, 0x41, 0x82, 0xE1, 0xC6,
	0x8A, 0x00, 0x7E, 0x5E, 0x0D, 0xD9, 0x02,
	0x0B, 0xFD, 0x64, 0xB6, 0x45, 0x03, 0x6C,
	0x7A, 0x4E, 0x67, 0x7D, 0x2C, 0x38, 0x53,
	0x2A, 0x3A, 0x23, 0xBA, 0x44, 0x42, 0xCA,
	0xF5, 0x3E, 0xA6, 0x3B, 0xB4, 0x54, 0x32,
	0x9B, 0x76, 0x24, 0xC8, 0x91, 0x7B, 0xDD,
	0x64, 0xB1, 0xC0, 0xFD, 0x4C, 0xB3, 0x8E,
	0x8C, 0x33, 0x4C, 0x70, 0x1C, 0x3A, 0xCD,
	0xAD, 0x06, 0x57, 0xFC, 0xCF, 0xEC, 0x71,
	0x9B, 0x1F, 0x5C, 0x3E, 0x4E, 0x46, 0x04,
	0x1F, 0x38, 0x81, 0x47, 0xFB, 0x4C, 0xFD,
	0xB4, 0x77, 0xA5, 0x24, 0x71, 0xF7, 0xA9,
	0xA9, 0x69, 0x10, 0xB8, 0x55, 0x32, 0x2E,
	0xDB, 0x63, 0x40, 0xD8, 0xA0, 0x0E, 0xF0,
	0x92, 0x35, 0x05, 0x11, 0xE3, 0x0A, 0xBE,
	0xC1, 0xFF, 0xF9, 0xE3, 0xA2, 0x6E, 0x7F,
	0xB2, 0x9F, 0x8C, 0x18, 0x30, 0x23, 0xC3,
	0x58, 0x7E, 0x38, 0xDA, 0x00, 0x77, 0xD9,
	0xB4, 0x76, 0x3E, 0x4E, 0x4B, 0x94, 0xB2,
	0xBB, 0xC1, 0x94, 0xC6, 0x65, 0x1E, 0x77,
	0xCA, 0xF9, 0x92, 0xEE, 0xAA, 0xC0, 0x23,
	0x2A, 0x28, 0x1B, 0xF6, 0xB3, 0xA7, 0x39,
	0xC1, 0x22, 0x61, 0x16, 0x82, 0x0A, 0xE8,
	0xDB, 0x58, 0x47, 0xA6, 0x7C, 0xBE, 0xF9,
	0xC9, 0x09, 0x1B, 0x46, 0x2D, 0x53, 0x8C,
	0xD7, 0x2B, 0x03, 0x74, 0x6A, 0xE7, 0x7F,
	0x5E, 0x62, 0x29, 0x2C, 0x31, 0x15, 0x62,
	0xA8, 0x46, 0x50, 0x5D, 0xC8, 0x2D, 0xB8,
	0x54, 0x33, 0x8A, 0xE4, 0x9F, 0x52, 0x35,
	0xC9, 0x5B, 0x91, 0x17, 0x8C, 0xCF, 0x2D,
	0xD5, 0xCA, 0xCE, 0xF4, 0x03, 0xEC, 0x9D,
	0x18, 0x10, 0xC6, 0x27, 0x2B, 0x04, 0x5B,
	0x3B, 0x71, 0xF9, 0xDC, 0x6B, 0x80, 0xD6,
	0x3F, 0xDD, 0x4A, 0x8E, 0x9A, 0xDB, 0x1E,
	0x69, 0x62, 0xA6, 0x95, 0x26, 0xD4, 0x31,
	0x61, 0xC1, 0xA4, 0x1D, 0x57, 0x0D, 0x79,
	0x38, 0xDA, 0xD4, 0xA4, 0x0E, 0x32, 0x9C,
	0xD0, 0xE4, 0x0E, 0x65, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

static const unsigned char ffdhe_q_6144[] = {
	0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xD6, 0xFC, 0x2A, 0x2C, 0x51, 0x5D,
	0xA5, 0x4D, 0x57, 0xEE, 0x2B, 0x10, 0x13,
	0x9E, 0x9E, 0x78, 0xEC, 0x5C, 0xE2, 0xC1,
	0xE7, 0x16, 0x9B, 0x4A, 0xD4, 0xF0, 0x9B,
	0x20, 0x8A, 0x32, 0x19, 0xFD, 0xE6, 0x49,
	0xCE, 0xE7, 0x12, 0x4D, 0x9F, 0x7C, 0xBE,
	0x97, 0xF1, 0xB1, 0xB1, 0x86, 0x3A, 0xEC,
	0x7B, 0x40, 0xD9, 0x01, 0x57, 0x62, 0x30,
	0xBD, 0x69, 0xEF, 0x8F, 0x6A, 0xEA, 0xFE,
	0xB2, 0xB0, 0x92, 0x19, 0xFA, 0x8F, 0xAF,
	0x83, 0x37, 0x68, 0x42, 0xB1, 0xB2, 0xAA,
	0x9E, 0xF6, 0x8D, 0x79, 0xDA, 0xAB, 0x89,
	0xAF, 0x3F, 0xAB, 0xE4, 0x9A, 0xCC, 0x27,
	0x86, 0x38, 0x70, 0x73, 0x45, 0xBB, 0xF1,
	0x53, 0x44, 0xED, 0x79, 0xF7, 0xF4, 0x39,
	0x0E, 0xF8, 0xAC, 0x50, 0x9B, 0x56, 0xF3,
	0x9A, 0x98, 0x56, 0x65, 0x27, 0xA4, 0x1D,
	0x3C, 0xBD, 0x5E, 0x05, 0x58, 0xC1, 0x59,
	0x92, 0x7D, 0xB0, 0xE8, 0x84, 0x54, 0xA5,
	0xD9, 0x64, 0x71, 0xFD, 0xDC, 0xB5, 0x6D,
	0x5B, 0xB0, 0x6B, 0xFA, 0x34, 0x0E, 0xA7,
	0xA1, 0x51, 0xEF, 0x1C, 0xA6, 0xFA, 0x57,
	0x2B, 0x76, 0xF3, 0xB1, 0xB9, 0x5D, 0x8C,
	0x85, 0x83, 0xD3, 0xE4, 0x77, 0x05, 0x36,
	0xB8, 0x4F, 0x01, 0x7E, 0x70, 0xE6, 0xFB,
	0xF1, 0x76, 0x60, 0x1A, 0x02, 0x66, 0x94,
	0x1A, 0x17, 0xB0, 0xC8, 0xB9, 0x7F, 0x4E,
	0x74, 0xC2, 0xC1, 0xFF, 0xC7, 0x27, 0x89,
	0x19, 0x77, 0x79, 0x40, 0xC1, 0xE1, 0xFF,
	0x1D, 0x8D, 0xA6, 0x37, 0xD6, 0xB9, 0x9D,
	0xDA, 0xFE, 0x5E, 0x17, 0x61, 0x10, 0x02,
	0xE2, 0xC7, 0x78, 0xC1, 0xBE, 0x8B, 0x41,
	0xD9, 0x63, 0x79, 0xA5, 0x13, 0x60, 0xD9,
	0x77, 0xFD, 0x44, 0x35, 0xA1, 0x1C, 0x30,
	0x8F, 0xE7, 0xEE, 0x6F, 0x1A, 0xAD, 0x9D,
	0xB2, 0x8C, 0x81, 0xAD, 0xDE, 0x1A, 0x7A,
	0x6F, 0x7C, 0xCE, 0x01, 0x1C, 0x30, 0xDA,
	0x37, 0xE4, 0xEB, 0x73, 0x64, 0x83, 0xBD,
	0x6C, 0x8E, 0x93, 0x48, 0xFB, 0xFB, 0xF7,
	0x2C, 0xC6, 0x58, 0x7D, 0x60, 0xC3, 0x6C,
	0x8E, 0x57, 0x7F, 0x09, 0x84, 0xC2, 0x89,
	0xC9, 0x38, 0x5A, 0x09, 0x86, 0x49, 0xDE,
	0x21, 0xBC, 0xA2, 0x7A, 0x7E, 0xA2, 0x29,
	0x71, 0x6B, 0xA6, 0xE9, 0xB2, 0x79, 0x71,
	0x0F, 0x38, 0xFA, 0xA5, 0xFF, 0xAE, 0x57,
	0x41, 0x55, 0xCE, 0x4E, 0xFB, 0x4F, 0x74,
	0x36, 0x95, 0xE2, 0x91, 0x1B, 0x1D, 0x06,
	0xD5, 0xE2, 0x90, 0xCB, 0xCD, 0x86, 0xF5,
	0x6D, 0x0E, 0xDF, 0xCD, 0x21, 0x6A, 0xE2,
	0x24, 0x27, 0x05, 0x5E, 0x68, 0x35, 0xFD,
	0x29, 0xEE, 0xF7, 0x9E, 0x0D, 0x90, 0x77,
	0x1F, 0xEA, 0xCE, 0xBE, 0x12, 0xF2, 0x0E,
	0x95, 0xB3, 0x4F, 0x0F, 0x78, 0xB7, 0x37,
	0xA9, 0x61, 0x8B, 0x26, 0xFA, 0x7D, 0xBC,
	0x98, 0x74, 0xF2, 0x72, 0xC4, 0x2B, 0xDB,
	0x56, 0x3E, 0xAF, 0xA1, 0x6B, 0x4F, 0xB6,
	0x8C, 0x3B, 0xB1, 0xE7, 0x8E, 0xAA, 0x81,
	0xA0, 0x02, 0x43, 0xFA, 0xAD, 0xD2, 0xBF,
	0x18, 0xE6, 0x3D, 0x38, 0x9A, 0xE4, 0x43,
	0x77, 0xDA, 0x18, 0xC5, 0x76, 0xB5, 0x0F,
	0x00, 0x96, 0xCF, 0x34, 0x19, 0x54, 0x83,
	0xB0, 0x05, 0x48, 0xC0, 0x98, 0x62, 0x36,
	0xE3, 0xBC, 0x7C, 0xB8, 0xD6, 0x80, 0x1C,
	0x04, 0x94, 0xCC, 0xD1, 0x99, 0xE5, 0xC5,
	0xBD, 0x0D, 0x0E, 0xDC, 0x9E, 0xB8, 0xA0,
	0x00, 0x1E, 0x15, 0x27, 0x67, 0x54, 0xFC,
	0xC6, 0x85, 0x66, 0x05, 0x41, 0x48, 0xE6,
	0xE7, 0x64, 0xBE, 0xE7, 0xC7, 0x64, 0xDA,
	0xAD, 0x3F, 0xC4, 0x52, 0x35, 0xA6, 0xDA,
	0xD4, 0x28, 0xFA, 0x20, 0xC1, 0x70, 0xE3,
	0x45, 0x00, 0x3F, 0x2F, 0x06, 0xEC, 0x81,
	0x05, 0xFE, 0xB2, 0x5B, 0x22, 0x81, 0xB6,
	0x3D, 0x27, 0x33, 0xBE, 0x96, 0x1C, 0x29,
	0x95, 0x1D, 0x11, 0xDD, 0x22, 0x21, 0x65,
	0x7A, 0x9F, 0x53, 0x1D, 0xDA, 0x2A, 0x19,
	0x4D, 0xBB, 0x12, 0x64, 0x48, 0xBD, 0xEE,
	0xB2, 0x58, 0xE0, 0x7E, 0xA6, 0x59, 0xC7,
	0x46, 0x19, 0xA6, 0x38, 0x0E, 0x1D, 0x66,
	0xD6, 0x83, 0x2B, 0xFE, 0x67, 0xF6, 0x38,
	0xCD, 0x8F, 0xAE, 0x1F, 0x27, 0x23, 0x02,
	0x0F, 0x9C, 0x40, 0xA3, 0xFD, 0xA6, 0x7E,
	0xDA, 0x3B, 0xD2, 0x92, 0x38, 0xFB, 0xD4,
	0xD4, 0xB4, 0x88, 0x5C, 0x2A, 0x99, 0x17,
	0x6D, 0xB1, 0xA0, 0x6C, 0x50, 0x07, 0x78,
	0x49, 0x1A, 0x82, 0x88, 0xF1, 0x85, 0x5F,
	0x60, 0xFF, 0xFC, 0xF1, 0xD1, 0x37, 0x3F,
	0xD9, 0x4F, 0xC6, 0x0C, 0x18, 0x11, 0xE1,
	0xAC, 0x3F, 0x1C, 0x6D, 0x00, 0x3B, 0xEC,
	0xDA, 0x3B, 0x1F, 0x27, 0x25, 0xCA, 0x59,
	0x5D, 0xE0, 0xCA, 0x63, 0x32, 0x8F, 0x3B,
	0xE5, 0x7C, 0xC9, 0x77, 0x55, 0x60, 0x11,
	0x95, 0x14, 0x0D, 0xFB, 0x59, 0xD3, 0x9C,
	0xE0, 0x91, 0x30, 0x8B, 0x41, 0x05, 0x74,
	0x6D, 0xAC, 0x23, 0xD3, 0x3E, 0x5F, 0x7C,
	0xE4, 0x84, 0x8D, 0xA3, 0x16, 0xA9, 0xC6,
	0x6B, 0x95, 0x81, 0xBA, 0x35, 0x73, 0xBF,
	0xAF, 0x31, 0x14, 0x96, 0x18, 0x8A, 0xB1,
	0x54, 0x23, 0x28, 0x2E, 0xE4, 0x16, 0xDC,
	0x2A, 0x19, 0xC5, 0x72, 0x4F, 0xA9, 0x1A,
	0xE4, 0xAD, 0xC8, 0x8B, 0xC6, 0x67, 0x96,
	0xEA, 0xE5, 0x67, 0x7A, 0x01, 0xF6, 0x4E,
	0x8C, 0x08, 0x63, 0x13, 0x95, 0x82, 0x2D,
	0x9D, 0xB8, 0xFC, 0xEE, 0x35, 0xC0, 0x6B,
	0x1F, 0xEE, 0xA5, 0x47, 0x4D, 0x6D, 0x8F,
	0x34, 0xB1, 0x53, 0x4A, 0x93, 0x6A, 0x18,
	0xB0, 0xE0, 0xD2, 0x0E, 0xAB, 0x86, 0xBC,
	0x9C, 0x6D, 0x6A, 0x52, 0x07, 0x19, 0x4E,
	0x68, 0x72, 0x07, 0x32, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

const gnutls_datum_t gnutls_ffdhe_6144_group_prime = {
	(void *) ffdhe_params_6144, sizeof(ffdhe_params_6144)
};
const gnutls_datum_t gnutls_ffdhe_6144_group_q = {
	(void *) ffdhe_q_6144, sizeof(ffdhe_q_6144)
};
const gnutls_datum_t gnutls_ffdhe_6144_group_generator = {
	(void *) &ffdhe_generator, sizeof(ffdhe_generator)
};
const unsigned int gnutls_ffdhe_6144_key_bits = 376;

static const unsigned char ffdhe_params_8192[] = {
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 
	0xFF, 0xAD, 0xF8, 0x54, 0x58, 0xA2, 0xBB, 
	0x4A, 0x9A, 0xAF, 0xDC, 0x56, 0x20, 0x27, 
	0x3D, 0x3C, 0xF1, 0xD8, 0xB9, 0xC5, 0x83, 
	0xCE, 0x2D, 0x36, 0x95, 0xA9, 0xE1, 0x36, 
	0x41, 0x14, 0x64, 0x33, 0xFB, 0xCC, 0x93, 
	0x9D, 0xCE, 0x24, 0x9B, 0x3E, 0xF9, 0x7D, 
	0x2F, 0xE3, 0x63, 0x63, 0x0C, 0x75, 0xD8, 
	0xF6, 0x81, 0xB2, 0x02, 0xAE, 0xC4, 0x61, 
	0x7A, 0xD3, 0xDF, 0x1E, 0xD5, 0xD5, 0xFD, 
	0x65, 0x61, 0x24, 0x33, 0xF5, 0x1F, 0x5F, 
	0x06, 0x6E, 0xD0, 0x85, 0x63, 0x65, 0x55, 
	0x3D, 0xED, 0x1A, 0xF3, 0xB5, 0x57, 0x13, 
	0x5E, 0x7F, 0x57, 0xC9, 0x35, 0x98, 0x4F, 
	0x0C, 0x70, 0xE0, 0xE6, 0x8B, 0x77, 0xE2, 
	0xA6, 0x89, 0xDA, 0xF3, 0xEF, 0xE8, 0x72, 
	0x1D, 0xF1, 0x58, 0xA1, 0x36, 0xAD, 0xE7, 
	0x35, 0x30, 0xAC, 0xCA, 0x4F, 0x48, 0x3A, 
	0x79, 0x7A, 0xBC, 0x0A, 0xB1, 0x82, 0xB3, 
	0x24, 0xFB, 0x61, 0xD1, 0x08, 0xA9, 0x4B, 
	0xB2, 0xC8, 0xE3, 0xFB, 0xB9, 0x6A, 0xDA, 
	0xB7, 0x60, 0xD7, 0xF4, 0x68, 0x1D, 0x4F, 
	0x42, 0xA3, 0xDE, 0x39, 0x4D, 0xF4, 0xAE, 
	0x56, 0xED, 0xE7, 0x63, 0x72, 0xBB, 0x19, 
	0x0B, 0x07, 0xA7, 0xC8, 0xEE, 0x0A, 0x6D, 
	0x70, 0x9E, 0x02, 0xFC, 0xE1, 0xCD, 0xF7, 
	0xE2, 0xEC, 0xC0, 0x34, 0x04, 0xCD, 0x28, 
	0x34, 0x2F, 0x61, 0x91, 0x72, 0xFE, 0x9C, 
	0xE9, 0x85, 0x83, 0xFF, 0x8E, 0x4F, 0x12, 
	0x32, 0xEE, 0xF2, 0x81, 0x83, 0xC3, 0xFE, 
	0x3B, 0x1B, 0x4C, 0x6F, 0xAD, 0x73, 0x3B, 
	0xB5, 0xFC, 0xBC, 0x2E, 0xC2, 0x20, 0x05, 
	0xC5, 0x8E, 0xF1, 0x83, 0x7D, 0x16, 0x83, 
	0xB2, 0xC6, 0xF3, 0x4A, 0x26, 0xC1, 0xB2, 
	0xEF, 0xFA, 0x88, 0x6B, 0x42, 0x38, 0x61, 
	0x1F, 0xCF, 0xDC, 0xDE, 0x35, 0x5B, 0x3B, 
	0x65, 0x19, 0x03, 0x5B, 0xBC, 0x34, 0xF4, 
	0xDE, 0xF9, 0x9C, 0x02, 0x38, 0x61, 0xB4, 
	0x6F, 0xC9, 0xD6, 0xE6, 0xC9, 0x07, 0x7A, 
	0xD9, 0x1D, 0x26, 0x91, 0xF7, 0xF7, 0xEE, 
	0x59, 0x8C, 0xB0, 0xFA, 0xC1, 0x86, 0xD9, 
	0x1C, 0xAE, 0xFE, 0x13, 0x09, 0x85, 0x13, 
	0x92, 0x70, 0xB4, 0x13, 0x0C, 0x93, 0xBC, 
	0x43, 0x79, 0x44, 0xF4, 0xFD, 0x44, 0x52, 
	0xE2, 0xD7, 0x4D, 0xD3, 0x64, 0xF2, 0xE2, 
	0x1E, 0x71, 0xF5, 0x4B, 0xFF, 0x5C, 0xAE, 
	0x82, 0xAB, 0x9C, 0x9D, 0xF6, 0x9E, 0xE8, 
	0x6D, 0x2B, 0xC5, 0x22, 0x36, 0x3A, 0x0D, 
	0xAB, 0xC5, 0x21, 0x97, 0x9B, 0x0D, 0xEA, 
	0xDA, 0x1D, 0xBF, 0x9A, 0x42, 0xD5, 0xC4, 
	0x48, 0x4E, 0x0A, 0xBC, 0xD0, 0x6B, 0xFA, 
	0x53, 0xDD, 0xEF, 0x3C, 0x1B, 0x20, 0xEE, 
	0x3F, 0xD5, 0x9D, 0x7C, 0x25, 0xE4, 0x1D, 
	0x2B, 0x66, 0x9E, 0x1E, 0xF1, 0x6E, 0x6F, 
	0x52, 0xC3, 0x16, 0x4D, 0xF4, 0xFB, 0x79, 
	0x30, 0xE9, 0xE4, 0xE5, 0x88, 0x57, 0xB6, 
	0xAC, 0x7D, 0x5F, 0x42, 0xD6, 0x9F, 0x6D, 
	0x18, 0x77, 0x63, 0xCF, 0x1D, 0x55, 0x03, 
	0x40, 0x04, 0x87, 0xF5, 0x5B, 0xA5, 0x7E, 
	0x31, 0xCC, 0x7A, 0x71, 0x35, 0xC8, 0x86, 
	0xEF, 0xB4, 0x31, 0x8A, 0xED, 0x6A, 0x1E, 
	0x01, 0x2D, 0x9E, 0x68, 0x32, 0xA9, 0x07, 
	0x60, 0x0A, 0x91, 0x81, 0x30, 0xC4, 0x6D, 
	0xC7, 0x78, 0xF9, 0x71, 0xAD, 0x00, 0x38, 
	0x09, 0x29, 0x99, 0xA3, 0x33, 0xCB, 0x8B, 
	0x7A, 0x1A, 0x1D, 0xB9, 0x3D, 0x71, 0x40, 
	0x00, 0x3C, 0x2A, 0x4E, 0xCE, 0xA9, 0xF9, 
	0x8D, 0x0A, 0xCC, 0x0A, 0x82, 0x91, 0xCD, 
	0xCE, 0xC9, 0x7D, 0xCF, 0x8E, 0xC9, 0xB5, 
	0x5A, 0x7F, 0x88, 0xA4, 0x6B, 0x4D, 0xB5, 
	0xA8, 0x51, 0xF4, 0x41, 0x82, 0xE1, 0xC6, 
	0x8A, 0x00, 0x7E, 0x5E, 0x0D, 0xD9, 0x02, 
	0x0B, 0xFD, 0x64, 0xB6, 0x45, 0x03, 0x6C, 
	0x7A, 0x4E, 0x67, 0x7D, 0x2C, 0x38, 0x53, 
	0x2A, 0x3A, 0x23, 0xBA, 0x44, 0x42, 0xCA, 
	0xF5, 0x3E, 0xA6, 0x3B, 0xB4, 0x54, 0x32, 
	0x9B, 0x76, 0x24, 0xC8, 0x91, 0x7B, 0xDD, 
	0x64, 0xB1, 0xC0, 0xFD, 0x4C, 0xB3, 0x8E, 
	0x8C, 0x33, 0x4C, 0x70, 0x1C, 0x3A, 0xCD, 
	0xAD, 0x06, 0x57, 0xFC, 0xCF, 0xEC, 0x71, 
	0x9B, 0x1F, 0x5C, 0x3E, 0x4E, 0x46, 0x04, 
	0x1F, 0x38, 0x81, 0x47, 0xFB, 0x4C, 0xFD, 
	0xB4, 0x77, 0xA5, 0x24, 0x71, 0xF7, 0xA9, 
	0xA9, 0x69, 0x10, 0xB8, 0x55, 0x32, 0x2E, 
	0xDB, 0x63, 0x40, 0xD8, 0xA0, 0x0E, 0xF0, 
	0x92, 0x35, 0x05, 0x11, 0xE3, 0x0A, 0xBE, 
	0xC1, 0xFF, 0xF9, 0xE3, 0xA2, 0x6E, 0x7F, 
	0xB2, 0x9F, 0x8C, 0x18, 0x30, 0x23, 0xC3, 
	0x58, 0x7E, 0x38, 0xDA, 0x00, 0x77, 0xD9, 
	0xB4, 0x76, 0x3E, 0x4E, 0x4B, 0x94, 0xB2, 
	0xBB, 0xC1, 0x94, 0xC6, 0x65, 0x1E, 0x77, 
	0xCA, 0xF9, 0x92, 0xEE, 0xAA, 0xC0, 0x23, 
	0x2A, 0x28, 0x1B, 0xF6, 0xB3, 0xA7, 0x39, 
	0xC1, 0x22, 0x61, 0x16, 0x82, 0x0A, 0xE8, 
	0xDB, 0x58, 0x47, 0xA6, 0x7C, 0xBE, 0xF9, 
	0xC9, 0x09, 0x1B, 0x46, 0x2D, 0x53, 0x8C, 
	0xD7, 0x2B, 0x03, 0x74, 0x6A, 0xE7, 0x7F, 
	0x5E, 0x62, 0x29, 0x2C, 0x31, 0x15, 0x62, 
	0xA8, 0x46, 0x50, 0x5D, 0xC8, 0x2D, 0xB8, 
	0x54, 0x33, 0x8A, 0xE4, 0x9F, 0x52, 0x35, 
	0xC9, 0x5B, 0x91, 0x17, 0x8C, 0xCF, 0x2D, 
	0xD5, 0xCA, 0xCE, 0xF4, 0x03, 0xEC, 0x9D, 
	0x18, 0x10, 0xC6, 0x27, 0x2B, 0x04, 0x5B, 
	0x3B, 0x71, 0xF9, 0xDC, 0x6B, 0x80, 0xD6, 
	0x3F, 0xDD, 0x4A, 0x8E, 0x9A, 0xDB, 0x1E, 
	0x69, 0x62, 0xA6, 0x95, 0x26, 0xD4, 0x31, 
	0x61, 0xC1, 0xA4, 0x1D, 0x57, 0x0D, 0x79, 
	0x38, 0xDA, 0xD4, 0xA4, 0x0E, 0x32, 0x9C, 
	0xCF, 0xF4, 0x6A, 0xAA, 0x36, 0xAD, 0x00, 
	0x4C, 0xF6, 0x00, 0xC8, 0x38, 0x1E, 0x42, 
	0x5A, 0x31, 0xD9, 0x51, 0xAE, 0x64, 0xFD, 
	0xB2, 0x3F, 0xCE, 0xC9, 0x50, 0x9D, 0x43, 
	0x68, 0x7F, 0xEB, 0x69, 0xED, 0xD1, 0xCC, 
	0x5E, 0x0B, 0x8C, 0xC3, 0xBD, 0xF6, 0x4B, 
	0x10, 0xEF, 0x86, 0xB6, 0x31, 0x42, 0xA3, 
	0xAB, 0x88, 0x29, 0x55, 0x5B, 0x2F, 0x74, 
	0x7C, 0x93, 0x26, 0x65, 0xCB, 0x2C, 0x0F, 
	0x1C, 0xC0, 0x1B, 0xD7, 0x02, 0x29, 0x38, 
	0x88, 0x39, 0xD2, 0xAF, 0x05, 0xE4, 0x54, 
	0x50, 0x4A, 0xC7, 0x8B, 0x75, 0x82, 0x82, 
	0x28, 0x46, 0xC0, 0xBA, 0x35, 0xC3, 0x5F, 
	0x5C, 0x59, 0x16, 0x0C, 0xC0, 0x46, 0xFD, 
	0x82, 0x51, 0x54, 0x1F, 0xC6, 0x8C, 0x9C, 
	0x86, 0xB0, 0x22, 0xBB, 0x70, 0x99, 0x87, 
	0x6A, 0x46, 0x0E, 0x74, 0x51, 0xA8, 0xA9, 
	0x31, 0x09, 0x70, 0x3F, 0xEE, 0x1C, 0x21, 
	0x7E, 0x6C, 0x38, 0x26, 0xE5, 0x2C, 0x51, 
	0xAA, 0x69, 0x1E, 0x0E, 0x42, 0x3C, 0xFC, 
	0x99, 0xE9, 0xE3, 0x16, 0x50, 0xC1, 0x21, 
	0x7B, 0x62, 0x48, 0x16, 0xCD, 0xAD, 0x9A, 
	0x95, 0xF9, 0xD5, 0xB8, 0x01, 0x94, 0x88, 
	0xD9, 0xC0, 0xA0, 0xA1, 0xFE, 0x30, 0x75, 
	0xA5, 0x77, 0xE2, 0x31, 0x83, 0xF8, 0x1D, 
	0x4A, 0x3F, 0x2F, 0xA4, 0x57, 0x1E, 0xFC, 
	0x8C, 0xE0, 0xBA, 0x8A, 0x4F, 0xE8, 0xB6, 
	0x85, 0x5D, 0xFE, 0x72, 0xB0, 0xA6, 0x6E, 
	0xDE, 0xD2, 0xFB, 0xAB, 0xFB, 0xE5, 0x8A, 
	0x30, 0xFA, 0xFA, 0xBE, 0x1C, 0x5D, 0x71, 
	0xA8, 0x7E, 0x2F, 0x74, 0x1E, 0xF8, 0xC1, 
	0xFE, 0x86, 0xFE, 0xA6, 0xBB, 0xFD, 0xE5, 
	0x30, 0x67, 0x7F, 0x0D, 0x97, 0xD1, 0x1D, 
	0x49, 0xF7, 0xA8, 0x44, 0x3D, 0x08, 0x22, 
	0xE5, 0x06, 0xA9, 0xF4, 0x61, 0x4E, 0x01, 
	0x1E, 0x2A, 0x94, 0x83, 0x8F, 0xF8, 0x8C, 
	0xD6, 0x8C, 0x8B, 0xB7, 0xC5, 0xC6, 0x42, 
	0x4C, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 
	0xFF, 0xFF
};

static const unsigned char ffdhe_q_8192[] = {
	0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xD6, 0xFC, 0x2A, 0x2C, 0x51, 0x5D,
	0xA5, 0x4D, 0x57, 0xEE, 0x2B, 0x10, 0x13,
	0x9E, 0x9E, 0x78, 0xEC, 0x5C, 0xE2, 0xC1,
	0xE7, 0x16, 0x9B, 0x4A, 0xD4, 0xF0, 0x9B,
	0x20, 0x8A, 0x32, 0x19, 0xFD, 0xE6, 0x49,
	0xCE, 0xE7, 0x12, 0x4D, 0x9F, 0x7C, 0xBE,
	0x97, 0xF1, 0xB1, 0xB1, 0x86, 0x3A, 0xEC,
	0x7B, 0x40, 0xD9, 0x01, 0x57, 0x62, 0x30,
	0xBD, 0x69, 0xEF, 0x8F, 0x6A, 0xEA, 0xFE,
	0xB2, 0xB0, 0x92, 0x19, 0xFA, 0x8F, 0xAF,
	0x83, 0x37, 0x68, 0x42, 0xB1, 0xB2, 0xAA,
	0x9E, 0xF6, 0x8D, 0x79, 0xDA, 0xAB, 0x89,
	0xAF, 0x3F, 0xAB, 0xE4, 0x9A, 0xCC, 0x27,
	0x86, 0x38, 0x70, 0x73, 0x45, 0xBB, 0xF1,
	0x53, 0x44, 0xED, 0x79, 0xF7, 0xF4, 0x39,
	0x0E, 0xF8, 0xAC, 0x50, 0x9B, 0x56, 0xF3,
	0x9A, 0x98, 0x56, 0x65, 0x27, 0xA4, 0x1D,
	0x3C, 0xBD, 0x5E, 0x05, 0x58, 0xC1, 0x59,
	0x92, 0x7D, 0xB0, 0xE8, 0x84, 0x54, 0xA5,
	0xD9, 0x64, 0x71, 0xFD, 0xDC, 0xB5, 0x6D,
	0x5B, 0xB0, 0x6B, 0xFA, 0x34, 0x0E, 0xA7,
	0xA1, 0x51, 0xEF, 0x1C, 0xA6, 0xFA, 0x57,
	0x2B, 0x76, 0xF3, 0xB1, 0xB9, 0x5D, 0x8C,
	0x85, 0x83, 0xD3, 0xE4, 0x77, 0x05, 0x36,
	0xB8, 0x4F, 0x01, 0x7E, 0x70, 0xE6, 0xFB,
	0xF1, 0x76, 0x60, 0x1A, 0x02, 0x66, 0x94,
	0x1A, 0x17, 0xB0, 0xC8, 0xB9, 0x7F, 0x4E,
	0x74, 0xC2, 0xC1, 0xFF, 0xC7, 0x27, 0x89,
	0x19, 0x77, 0x79, 0x40, 0xC1, 0xE1, 0xFF,
	0x1D, 0x8D, 0xA6, 0x37, 0xD6, 0xB9, 0x9D,
	0xDA, 0xFE, 0x5E, 0x17, 0x61, 0x10, 0x02,
	0xE2, 0xC7, 0x78, 0xC1, 0xBE, 0x8B, 0x41,
	0xD9, 0x63, 0x79, 0xA5, 0x13, 0x60, 0xD9,
	0x77, 0xFD, 0x44, 0x35, 0xA1, 0x1C, 0x30,
	0x8F, 0xE7, 0xEE, 0x6F, 0x1A, 0xAD, 0x9D,
	0xB2, 0x8C, 0x81, 0xAD, 0xDE, 0x1A, 0x7A,
	0x6F, 0x7C, 0xCE, 0x01, 0x1C, 0x30, 0xDA,
	0x37, 0xE4, 0xEB, 0x73, 0x64, 0x83, 0xBD,
	0x6C, 0x8E, 0x93, 0x48, 0xFB, 0xFB, 0xF7,
	0x2C, 0xC6, 0x58, 0x7D, 0x60, 0xC3, 0x6C,
	0x8E, 0x57, 0x7F, 0x09, 0x84, 0xC2, 0x89,
	0xC9, 0x38, 0x5A, 0x09, 0x86, 0x49, 0xDE,
	0x21, 0xBC, 0xA2, 0x7A, 0x7E, 0xA2, 0x29,
	0x71, 0x6B, 0xA6, 0xE9, 0xB2, 0x79, 0x71,
	0x0F, 0x38, 0xFA, 0xA5, 0xFF, 0xAE, 0x57,
	0x41, 0x55, 0xCE, 0x4E, 0xFB, 0x4F, 0x74,
	0x36, 0x95, 0xE2, 0x91, 0x1B, 0x1D, 0x06,
	0xD5, 0xE2, 0x90, 0xCB, 0xCD, 0x86, 0xF5,
	0x6D, 0x0E, 0xDF, 0xCD, 0x21, 0x6A, 0xE2,
	0x24, 0x27, 0x05, 0x5E, 0x68, 0x35, 0xFD,
	0x29, 0xEE, 0xF7, 0x9E, 0x0D, 0x90, 0x77,
	0x1F, 0xEA, 0xCE, 0xBE, 0x12, 0xF2, 0x0E,
	0x95, 0xB3, 0x4F, 0x0F, 0x78, 0xB7, 0x37,
	0xA9, 0x61, 0x8B, 0x26, 0xFA, 0x7D, 0xBC,
	0x98, 0x74, 0xF2, 0x72, 0xC4, 0x2B, 0xDB,
	0x56, 0x3E, 0xAF, 0xA1, 0x6B, 0x4F, 0xB6,
	0x8C, 0x3B, 0xB1, 0xE7, 0x8E, 0xAA, 0x81,
	0xA0, 0x02, 0x43, 0xFA, 0xAD, 0xD2, 0xBF,
	0x18, 0xE6, 0x3D, 0x38, 0x9A, 0xE4, 0x43,
	0x77, 0xDA, 0x18, 0xC5, 0x76, 0xB5, 0x0F,
	0x00, 0x96, 0xCF, 0x34, 0x19, 0x54, 0x83,
	0xB0, 0x05, 0x48, 0xC0, 0x98, 0x62, 0x36,
	0xE3, 0xBC, 0x7C, 0xB8, 0xD6, 0x80, 0x1C,
	0x04, 0x94, 0xCC, 0xD1, 0x99, 0xE5, 0xC5,
	0xBD, 0x0D, 0x0E, 0xDC, 0x9E, 0xB8, 0xA0,
	0x00, 0x1E, 0x15, 0x27, 0x67, 0x54, 0xFC,
	0xC6, 0x85, 0x66, 0x05, 0x41, 0x48, 0xE6,
	0xE7, 0x64, 0xBE, 0xE7, 0xC7, 0x64, 0xDA,
	0xAD, 0x3F, 0xC4, 0x52, 0x35, 0xA6, 0xDA,
	0xD4, 0x28, 0xFA, 0x20, 0xC1, 0x70, 0xE3,
	0x45, 0x00, 0x3F, 0x2F, 0x06, 0xEC, 0x81,
	0x05, 0xFE, 0xB2, 0x5B, 0x22, 0x81, 0xB6,
	0x3D, 0x27, 0x33, 0xBE, 0x96, 0x1C, 0x29,
	0x95, 0x1D, 0x11, 0xDD, 0x22, 0x21, 0x65,
	0x7A, 0x9F, 0x53, 0x1D, 0xDA, 0x2A, 0x19,
	0x4D, 0xBB, 0x12, 0x64, 0x48, 0xBD, 0xEE,
	0xB2, 0x58, 0xE0, 0x7E, 0xA6, 0x59, 0xC7,
	0x46, 0x19, 0xA6, 0x38, 0x0E, 0x1D, 0x66,
	0xD6, 0x83, 0x2B, 0xFE, 0x67, 0xF6, 0x38,
	0xCD, 0x8F, 0xAE, 0x1F, 0x27, 0x23, 0x02,
	0x0F, 0x9C, 0x40, 0xA3, 0xFD, 0xA6, 0x7E,
	0xDA, 0x3B, 0xD2, 0x92, 0x38, 0xFB, 0xD4,
	0xD4, 0xB4, 0x88, 0x5C, 0x2A, 0x99, 0x17,
	0x6D, 0xB1, 0xA0, 0x6C, 0x50, 0x07, 0x78,
	0x49, 0x1A, 0x82, 0x88, 0xF1, 0x85, 0x5F,
	0x60, 0xFF, 0xFC, 0xF1, 0xD1, 0x37, 0x3F,
	0xD9, 0x4F, 0xC6, 0x0C, 0x18, 0x11, 0xE1,
	0xAC, 0x3F, 0x1C, 0x6D, 0x00, 0x3B, 0xEC,
	0xDA, 0x3B, 0x1F, 0x27, 0x25, 0xCA, 0x59,
	0x5D, 0xE0, 0xCA, 0x63, 0x32, 0x8F, 0x3B,
	0xE5, 0x7C, 0xC9, 0x77, 0x55, 0x60, 0x11,
	0x95, 0x14, 0x0D, 0xFB, 0x59, 0xD3, 0x9C,
	0xE0, 0x91, 0x30, 0x8B, 0x41, 0x05, 0x74,
	0x6D, 0xAC, 0x23, 0xD3, 0x3E, 0x5F, 0x7C,
	0xE4, 0x84, 0x8D, 0xA3, 0x16, 0xA9, 0xC6,
	0x6B, 0x95, 0x81, 0xBA, 0x35, 0x73, 0xBF,
	0xAF, 0x31, 0x14, 0x96, 0x18, 0x8A, 0xB1,
	0x54, 0x23, 0x28, 0x2E, 0xE4, 0x16, 0xDC,
	0x2A, 0x19, 0xC5, 0x72, 0x4F, 0xA9, 0x1A,
	0xE4, 0xAD, 0xC8, 0x8B, 0xC6, 0x67, 0x96,
	0xEA, 0xE5, 0x67, 0x7A, 0x01, 0xF6, 0x4E,
	0x8C, 0x08, 0x63, 0x13, 0x95, 0x82, 0x2D,
	0x9D, 0xB8, 0xFC, 0xEE, 0x35, 0xC0, 0x6B,
	0x1F, 0xEE, 0xA5, 0x47, 0x4D, 0x6D, 0x8F,
	0x34, 0xB1, 0x53, 0x4A, 0x93, 0x6A, 0x18,
	0xB0, 0xE0, 0xD2, 0x0E, 0xAB, 0x86, 0xBC,
	0x9C, 0x6D, 0x6A, 0x52, 0x07, 0x19, 0x4E,
	0x67, 0xFA, 0x35, 0x55, 0x1B, 0x56, 0x80,
	0x26, 0x7B, 0x00, 0x64, 0x1C, 0x0F, 0x21,
	0x2D, 0x18, 0xEC, 0xA8, 0xD7, 0x32, 0x7E,
	0xD9, 0x1F, 0xE7, 0x64, 0xA8, 0x4E, 0xA1,
	0xB4, 0x3F, 0xF5, 0xB4, 0xF6, 0xE8, 0xE6,
	0x2F, 0x05, 0xC6, 0x61, 0xDE, 0xFB, 0x25,
	0x88, 0x77, 0xC3, 0x5B, 0x18, 0xA1, 0x51,
	0xD5, 0xC4, 0x14, 0xAA, 0xAD, 0x97, 0xBA,
	0x3E, 0x49, 0x93, 0x32, 0xE5, 0x96, 0x07,
	0x8E, 0x60, 0x0D, 0xEB, 0x81, 0x14, 0x9C,
	0x44, 0x1C, 0xE9, 0x57, 0x82, 0xF2, 0x2A,
	0x28, 0x25, 0x63, 0xC5, 0xBA, 0xC1, 0x41,
	0x14, 0x23, 0x60, 0x5D, 0x1A, 0xE1, 0xAF,
	0xAE, 0x2C, 0x8B, 0x06, 0x60, 0x23, 0x7E,
	0xC1, 0x28, 0xAA, 0x0F, 0xE3, 0x46, 0x4E,
	0x43, 0x58, 0x11, 0x5D, 0xB8, 0x4C, 0xC3,
	0xB5, 0x23, 0x07, 0x3A, 0x28, 0xD4, 0x54,
	0x98, 0x84, 0xB8, 0x1F, 0xF7, 0x0E, 0x10,
	0xBF, 0x36, 0x1C, 0x13, 0x72, 0x96, 0x28,
	0xD5, 0x34, 0x8F, 0x07, 0x21, 0x1E, 0x7E,
	0x4C, 0xF4, 0xF1, 0x8B, 0x28, 0x60, 0x90,
	0xBD, 0xB1, 0x24, 0x0B, 0x66, 0xD6, 0xCD,
	0x4A, 0xFC, 0xEA, 0xDC, 0x00, 0xCA, 0x44,
	0x6C, 0xE0, 0x50, 0x50, 0xFF, 0x18, 0x3A,
	0xD2, 0xBB, 0xF1, 0x18, 0xC1, 0xFC, 0x0E,
	0xA5, 0x1F, 0x97, 0xD2, 0x2B, 0x8F, 0x7E,
	0x46, 0x70, 0x5D, 0x45, 0x27, 0xF4, 0x5B,
	0x42, 0xAE, 0xFF, 0x39, 0x58, 0x53, 0x37,
	0x6F, 0x69, 0x7D, 0xD5, 0xFD, 0xF2, 0xC5,
	0x18, 0x7D, 0x7D, 0x5F, 0x0E, 0x2E, 0xB8,
	0xD4, 0x3F, 0x17, 0xBA, 0x0F, 0x7C, 0x60,
	0xFF, 0x43, 0x7F, 0x53, 0x5D, 0xFE, 0xF2,
	0x98, 0x33, 0xBF, 0x86, 0xCB, 0xE8, 0x8E,
	0xA4, 0xFB, 0xD4, 0x22, 0x1E, 0x84, 0x11,
	0x72, 0x83, 0x54, 0xFA, 0x30, 0xA7, 0x00,
	0x8F, 0x15, 0x4A, 0x41, 0xC7, 0xFC, 0x46,
	0x6B, 0x46, 0x45, 0xDB, 0xE2, 0xE3, 0x21,
	0x26, 0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF
};

const gnutls_datum_t gnutls_ffdhe_8192_group_prime = {
	(void *) ffdhe_params_8192, sizeof(ffdhe_params_8192)
};
const gnutls_datum_t gnutls_ffdhe_8192_group_q = {
	(void *) ffdhe_q_8192, sizeof(ffdhe_q_8192)
};
const gnutls_datum_t gnutls_ffdhe_8192_group_generator = {
	(void *) &ffdhe_generator, sizeof(ffdhe_generator)
};
const unsigned int gnutls_ffdhe_8192_key_bits = 512;

static const unsigned char modp_generator = 0x02;

static const unsigned char modp_params_2048[] = {
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xC9, 0x0F, 0xDA, 0xA2, 0x21, 0x68,
	0xC2, 0x34, 0xC4, 0xC6, 0x62, 0x8B, 0x80,
	0xDC, 0x1C, 0xD1, 0x29, 0x02, 0x4E, 0x08,
	0x8A, 0x67, 0xCC, 0x74, 0x02, 0x0B, 0xBE,
	0xA6, 0x3B, 0x13, 0x9B, 0x22, 0x51, 0x4A,
	0x08, 0x79, 0x8E, 0x34, 0x04, 0xDD, 0xEF,
	0x95, 0x19, 0xB3, 0xCD, 0x3A, 0x43, 0x1B,
	0x30, 0x2B, 0x0A, 0x6D, 0xF2, 0x5F, 0x14,
	0x37, 0x4F, 0xE1, 0x35, 0x6D, 0x6D, 0x51,
	0xC2, 0x45, 0xE4, 0x85, 0xB5, 0x76, 0x62,
	0x5E, 0x7E, 0xC6, 0xF4, 0x4C, 0x42, 0xE9,
	0xA6, 0x37, 0xED, 0x6B, 0x0B, 0xFF, 0x5C,
	0xB6, 0xF4, 0x06, 0xB7, 0xED, 0xEE, 0x38,
	0x6B, 0xFB, 0x5A, 0x89, 0x9F, 0xA5, 0xAE,
	0x9F, 0x24, 0x11, 0x7C, 0x4B, 0x1F, 0xE6,
	0x49, 0x28, 0x66, 0x51, 0xEC, 0xE4, 0x5B,
	0x3D, 0xC2, 0x00, 0x7C, 0xB8, 0xA1, 0x63,
	0xBF, 0x05, 0x98, 0xDA, 0x48, 0x36, 0x1C,
	0x55, 0xD3, 0x9A, 0x69, 0x16, 0x3F, 0xA8,
	0xFD, 0x24, 0xCF, 0x5F, 0x83, 0x65, 0x5D,
	0x23, 0xDC, 0xA3, 0xAD, 0x96, 0x1C, 0x62,
	0xF3, 0x56, 0x20, 0x85, 0x52, 0xBB, 0x9E,
	0xD5, 0x29, 0x07, 0x70, 0x96, 0x96, 0x6D,
	0x67, 0x0C, 0x35, 0x4E, 0x4A, 0xBC, 0x98,
	0x04, 0xF1, 0x74, 0x6C, 0x08, 0xCA, 0x18,
	0x21, 0x7C, 0x32, 0x90, 0x5E, 0x46, 0x2E,
	0x36, 0xCE, 0x3B, 0xE3, 0x9E, 0x77, 0x2C,
	0x18, 0x0E, 0x86, 0x03, 0x9B, 0x27, 0x83,
	0xA2, 0xEC, 0x07, 0xA2, 0x8F, 0xB5, 0xC5,
	0x5D, 0xF0, 0x6F, 0x4C, 0x52, 0xC9, 0xDE,
	0x2B, 0xCB, 0xF6, 0x95, 0x58, 0x17, 0x18,
	0x39, 0x95, 0x49, 0x7C, 0xEA, 0x95, 0x6A,
	0xE5, 0x15, 0xD2, 0x26, 0x18, 0x98, 0xFA,
	0x05, 0x10, 0x15, 0x72, 0x8E, 0x5A, 0x8A,
	0xAC, 0xAA, 0x68, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF
};

static const unsigned char modp_q_2048[] = {
	0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xE4, 0x87, 0xED, 0x51, 0x10, 0xB4,
	0x61, 0x1A, 0x62, 0x63, 0x31, 0x45, 0xC0,
	0x6E, 0x0E, 0x68, 0x94, 0x81, 0x27, 0x04,
	0x45, 0x33, 0xE6, 0x3A, 0x01, 0x05, 0xDF,
	0x53, 0x1D, 0x89, 0xCD, 0x91, 0x28, 0xA5,
	0x04, 0x3C, 0xC7, 0x1A, 0x02, 0x6E, 0xF7,
	0xCA, 0x8C, 0xD9, 0xE6, 0x9D, 0x21, 0x8D,
	0x98, 0x15, 0x85, 0x36, 0xF9, 0x2F, 0x8A,
	0x1B, 0xA7, 0xF0, 0x9A, 0xB6, 0xB6, 0xA8,
	0xE1, 0x22, 0xF2, 0x42, 0xDA, 0xBB, 0x31,
	0x2F, 0x3F, 0x63, 0x7A, 0x26, 0x21, 0x74,
	0xD3, 0x1B, 0xF6, 0xB5, 0x85, 0xFF, 0xAE,
	0x5B, 0x7A, 0x03, 0x5B, 0xF6, 0xF7, 0x1C,
	0x35, 0xFD, 0xAD, 0x44, 0xCF, 0xD2, 0xD7,
	0x4F, 0x92, 0x08, 0xBE, 0x25, 0x8F, 0xF3,
	0x24, 0x94, 0x33, 0x28, 0xF6, 0x72, 0x2D,
	0x9E, 0xE1, 0x00, 0x3E, 0x5C, 0x50, 0xB1,
	0xDF, 0x82, 0xCC, 0x6D, 0x24, 0x1B, 0x0E,
	0x2A, 0xE9, 0xCD, 0x34, 0x8B, 0x1F, 0xD4,
	0x7E, 0x92, 0x67, 0xAF, 0xC1, 0xB2, 0xAE,
	0x91, 0xEE, 0x51, 0xD6, 0xCB, 0x0E, 0x31,
	0x79, 0xAB, 0x10, 0x42, 0xA9, 0x5D, 0xCF,
	0x6A, 0x94, 0x83, 0xB8, 0x4B, 0x4B, 0x36,
	0xB3, 0x86, 0x1A, 0xA7, 0x25, 0x5E, 0x4C,
	0x02, 0x78, 0xBA, 0x36, 0x04, 0x65, 0x0C,
	0x10, 0xBE, 0x19, 0x48, 0x2F, 0x23, 0x17,
	0x1B, 0x67, 0x1D, 0xF1, 0xCF, 0x3B, 0x96,
	0x0C, 0x07, 0x43, 0x01, 0xCD, 0x93, 0xC1,
	0xD1, 0x76, 0x03, 0xD1, 0x47, 0xDA, 0xE2,
	0xAE, 0xF8, 0x37, 0xA6, 0x29, 0x64, 0xEF,
	0x15, 0xE5, 0xFB, 0x4A, 0xAC, 0x0B, 0x8C,
	0x1C, 0xCA, 0xA4, 0xBE, 0x75, 0x4A, 0xB5,
	0x72, 0x8A, 0xE9, 0x13, 0x0C, 0x4C, 0x7D,
	0x02, 0x88, 0x0A, 0xB9, 0x47, 0x2D, 0x45,
	0x56, 0x55, 0x34, 0x7F, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF
};

const gnutls_datum_t gnutls_modp_2048_group_prime = {
	(void *) modp_params_2048, sizeof(modp_params_2048)
};
const gnutls_datum_t gnutls_modp_2048_group_q = {
	(void *) modp_q_2048, sizeof(modp_q_2048)
};
const gnutls_datum_t gnutls_modp_2048_group_generator = {
	(void *) &modp_generator, sizeof(modp_generator)
};
const unsigned int gnutls_modp_2048_key_bits = 256;

static const unsigned char modp_params_3072[] = {
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xC9, 0x0F, 0xDA, 0xA2, 0x21, 0x68,
	0xC2, 0x34, 0xC4, 0xC6, 0x62, 0x8B, 0x80,
	0xDC, 0x1C, 0xD1, 0x29, 0x02, 0x4E, 0x08,
	0x8A, 0x67, 0xCC, 0x74, 0x02, 0x0B, 0xBE,
	0xA6, 0x3B, 0x13, 0x9B, 0x22, 0x51, 0x4A,
	0x08, 0x79, 0x8E, 0x34, 0x04, 0xDD, 0xEF,
	0x95, 0x19, 0xB3, 0xCD, 0x3A, 0x43, 0x1B,
	0x30, 0x2B, 0x0A, 0x6D, 0xF2, 0x5F, 0x14,
	0x37, 0x4F, 0xE1, 0x35, 0x6D, 0x6D, 0x51,
	0xC2, 0x45, 0xE4, 0x85, 0xB5, 0x76, 0x62,
	0x5E, 0x7E, 0xC6, 0xF4, 0x4C, 0x42, 0xE9,
	0xA6, 0x37, 0xED, 0x6B, 0x0B, 0xFF, 0x5C,
	0xB6, 0xF4, 0x06, 0xB7, 0xED, 0xEE, 0x38,
	0x6B, 0xFB, 0x5A, 0x89, 0x9F, 0xA5, 0xAE,
	0x9F, 0x24, 0x11, 0x7C, 0x4B, 0x1F, 0xE6,
	0x49, 0x28, 0x66, 0x51, 0xEC, 0xE4, 0x5B,
	0x3D, 0xC2, 0x00, 0x7C, 0xB8, 0xA1, 0x63,
	0xBF, 0x05, 0x98, 0xDA, 0x48, 0x36, 0x1C,
	0x55, 0xD3, 0x9A, 0x69, 0x16, 0x3F, 0xA8,
	0xFD, 0x24, 0xCF, 0x5F, 0x83, 0x65, 0x5D,
	0x23, 0xDC, 0xA3, 0xAD, 0x96, 0x1C, 0x62,
	0xF3, 0x56, 0x20, 0x85, 0x52, 0xBB, 0x9E,
	0xD5, 0x29, 0x07, 0x70, 0x96, 0x96, 0x6D,
	0x67, 0x0C, 0x35, 0x4E, 0x4A, 0xBC, 0x98,
	0x04, 0xF1, 0x74, 0x6C, 0x08, 0xCA, 0x18,
	0x21, 0x7C, 0x32, 0x90, 0x5E, 0x46, 0x2E,
	0x36, 0xCE, 0x3B, 0xE3, 0x9E, 0x77, 0x2C,
	0x18, 0x0E, 0x86, 0x03, 0x9B, 0x27, 0x83,
	0xA2, 0xEC, 0x07, 0xA2, 0x8F, 0xB5, 0xC5,
	0x5D, 0xF0, 0x6F, 0x4C, 0x52, 0xC9, 0xDE,
	0x2B, 0xCB, 0xF6, 0x95, 0x58, 0x17, 0x18,
	0x39, 0x95, 0x49, 0x7C, 0xEA, 0x95, 0x6A,
	0xE5, 0x15, 0xD2, 0x26, 0x18, 0x98, 0xFA,
	0x05, 0x10, 0x15, 0x72, 0x8E, 0x5A, 0x8A,
	0xAA, 0xC4, 0x2D, 0xAD, 0x33, 0x17, 0x0D,
	0x04, 0x50, 0x7A, 0x33, 0xA8, 0x55, 0x21,
	0xAB, 0xDF, 0x1C, 0xBA, 0x64, 0xEC, 0xFB,
	0x85, 0x04, 0x58, 0xDB, 0xEF, 0x0A, 0x8A,
	0xEA, 0x71, 0x57, 0x5D, 0x06, 0x0C, 0x7D,
	0xB3, 0x97, 0x0F, 0x85, 0xA6, 0xE1, 0xE4,
	0xC7, 0xAB, 0xF5, 0xAE, 0x8C, 0xDB, 0x09,
	0x33, 0xD7, 0x1E, 0x8C, 0x94, 0xE0, 0x4A,
	0x25, 0x61, 0x9D, 0xCE, 0xE3, 0xD2, 0x26,
	0x1A, 0xD2, 0xEE, 0x6B, 0xF1, 0x2F, 0xFA,
	0x06, 0xD9, 0x8A, 0x08, 0x64, 0xD8, 0x76,
	0x02, 0x73, 0x3E, 0xC8, 0x6A, 0x64, 0x52,
	0x1F, 0x2B, 0x18, 0x17, 0x7B, 0x20, 0x0C,
	0xBB, 0xE1, 0x17, 0x57, 0x7A, 0x61, 0x5D,
	0x6C, 0x77, 0x09, 0x88, 0xC0, 0xBA, 0xD9,
	0x46, 0xE2, 0x08, 0xE2, 0x4F, 0xA0, 0x74,
	0xE5, 0xAB, 0x31, 0x43, 0xDB, 0x5B, 0xFC,
	0xE0, 0xFD, 0x10, 0x8E, 0x4B, 0x82, 0xD1,
	0x20, 0xA9, 0x3A, 0xD2, 0xCA, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

static const unsigned char modp_q_3072[] = {
	0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xE4, 0x87, 0xED, 0x51, 0x10, 0xB4,
	0x61, 0x1A, 0x62, 0x63, 0x31, 0x45, 0xC0,
	0x6E, 0x0E, 0x68, 0x94, 0x81, 0x27, 0x04,
	0x45, 0x33, 0xE6, 0x3A, 0x01, 0x05, 0xDF,
	0x53, 0x1D, 0x89, 0xCD, 0x91, 0x28, 0xA5,
	0x04, 0x3C, 0xC7, 0x1A, 0x02, 0x6E, 0xF7,
	0xCA, 0x8C, 0xD9, 0xE6, 0x9D, 0x21, 0x8D,
	0x98, 0x15, 0x85, 0x36, 0xF9, 0x2F, 0x8A,
	0x1B, 0xA7, 0xF0, 0x9A, 0xB6, 0xB6, 0xA8,
	0xE1, 0x22, 0xF2, 0x42, 0xDA, 0xBB, 0x31,
	0x2F, 0x3F, 0x63, 0x7A, 0x26, 0x21, 0x74,
	0xD3, 0x1B, 0xF6, 0xB5, 0x85, 0xFF, 0xAE,
	0x5B, 0x7A, 0x03, 0x5B, 0xF6, 0xF7, 0x1C,
	0x35, 0xFD, 0xAD, 0x44, 0xCF, 0xD2, 0xD7,
	0x4F, 0x92, 0x08, 0xBE, 0x25, 0x8F, 0xF3,
	0x24, 0x94, 0x33, 0x28, 0xF6, 0x72, 0x2D,
	0x9E, 0xE1, 0x00, 0x3E, 0x5C, 0x50, 0xB1,
	0xDF, 0x82, 0xCC, 0x6D, 0x24, 0x1B, 0x0E,
	0x2A, 0xE9, 0xCD, 0x34, 0x8B, 0x1F, 0xD4,
	0x7E, 0x92, 0x67, 0xAF, 0xC1, 0xB2, 0xAE,
	0x91, 0xEE, 0x51, 0xD6, 0xCB, 0x0E, 0x31,
	0x79, 0xAB, 0x10, 0x42, 0xA9, 0x5D, 0xCF,
	0x6A, 0x94, 0x83, 0xB8, 0x4B, 0x4B, 0x36,
	0xB3, 0x86, 0x1A, 0xA7, 0x25, 0x5E, 0x4C,
	0x02, 0x78, 0xBA, 0x36, 0x04, 0x65, 0x0C,
	0x10, 0xBE, 0x19, 0x48, 0x2F, 0x23, 0x17,
	0x1B, 0x67, 0x1D, 0xF1, 0xCF, 0x3B, 0x96,
	0x0C, 0x07, 0x43, 0x01, 0xCD, 0x93, 0xC1,
	0xD1, 0x76, 0x03, 0xD1, 0x47, 0xDA, 0xE2,
	0xAE, 0xF8, 0x37, 0xA6, 0x29, 0x64, 0xEF,
	0x15, 0xE5, 0xFB, 0x4A, 0xAC, 0x0B, 0x8C,
	0x1C, 0xCA, 0xA4, 0xBE, 0x75, 0x4A, 0xB5,
	0x72, 0x8A, 0xE9, 0x13, 0x0C, 0x4C, 0x7D,
	0x02, 0x88, 0x0A, 0xB9, 0x47, 0x2D, 0x45,
	0x55, 0x62, 0x16, 0xD6, 0x99, 0x8B, 0x86,
	0x82, 0x28, 0x3D, 0x19, 0xD4, 0x2A, 0x90,
	0xD5, 0xEF, 0x8E, 0x5D, 0x32, 0x76, 0x7D,
	0xC2, 0x82, 0x2C, 0x6D, 0xF7, 0x85, 0x45,
	0x75, 0x38, 0xAB, 0xAE, 0x83, 0x06, 0x3E,
	0xD9, 0xCB, 0x87, 0xC2, 0xD3, 0x70, 0xF2,
	0x63, 0xD5, 0xFA, 0xD7, 0x46, 0x6D, 0x84,
	0x99, 0xEB, 0x8F, 0x46, 0x4A, 0x70, 0x25,
	0x12, 0xB0, 0xCE, 0xE7, 0x71, 0xE9, 0x13,
	0x0D, 0x69, 0x77, 0x35, 0xF8, 0x97, 0xFD,
	0x03, 0x6C, 0xC5, 0x04, 0x32, 0x6C, 0x3B,
	0x01, 0x39, 0x9F, 0x64, 0x35, 0x32, 0x29,
	0x0F, 0x95, 0x8C, 0x0B, 0xBD, 0x90, 0x06,
	0x5D, 0xF0, 0x8B, 0xAB, 0xBD, 0x30, 0xAE,
	0xB6, 0x3B, 0x84, 0xC4, 0x60, 0x5D, 0x6C,
	0xA3, 0x71, 0x04, 0x71, 0x27, 0xD0, 0x3A,
	0x72, 0xD5, 0x98, 0xA1, 0xED, 0xAD, 0xFE,
	0x70, 0x7E, 0x88, 0x47, 0x25, 0xC1, 0x68,
	0x90, 0x54, 0x9D, 0x69, 0x65, 0x7F, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

const gnutls_datum_t gnutls_modp_3072_group_prime = {
	(void *) modp_params_3072, sizeof(modp_params_3072)
};
const gnutls_datum_t gnutls_modp_3072_group_q = {
	(void *) modp_q_3072, sizeof(modp_q_3072)
};
const gnutls_datum_t gnutls_modp_3072_group_generator = {
	(void *) &modp_generator, sizeof(modp_generator)
};
const unsigned int gnutls_modp_3072_key_bits = 276;

static const unsigned char modp_params_4096[] = {
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xC9, 0x0F, 0xDA, 0xA2, 0x21, 0x68,
	0xC2, 0x34, 0xC4, 0xC6, 0x62, 0x8B, 0x80,
	0xDC, 0x1C, 0xD1, 0x29, 0x02, 0x4E, 0x08,
	0x8A, 0x67, 0xCC, 0x74, 0x02, 0x0B, 0xBE,
	0xA6, 0x3B, 0x13, 0x9B, 0x22, 0x51, 0x4A,
	0x08, 0x79, 0x8E, 0x34, 0x04, 0xDD, 0xEF,
	0x95, 0x19, 0xB3, 0xCD, 0x3A, 0x43, 0x1B,
	0x30, 0x2B, 0x0A, 0x6D, 0xF2, 0x5F, 0x14,
	0x37, 0x4F, 0xE1, 0x35, 0x6D, 0x6D, 0x51,
	0xC2, 0x45, 0xE4, 0x85, 0xB5, 0x76, 0x62,
	0x5E, 0x7E, 0xC6, 0xF4, 0x4C, 0x42, 0xE9,
	0xA6, 0x37, 0xED, 0x6B, 0x0B, 0xFF, 0x5C,
	0xB6, 0xF4, 0x06, 0xB7, 0xED, 0xEE, 0x38,
	0x6B, 0xFB, 0x5A, 0x89, 0x9F, 0xA5, 0xAE,
	0x9F, 0x24, 0x11, 0x7C, 0x4B, 0x1F, 0xE6,
	0x49, 0x28, 0x66, 0x51, 0xEC, 0xE4, 0x5B,
	0x3D, 0xC2, 0x00, 0x7C, 0xB8, 0xA1, 0x63,
	0xBF, 0x05, 0x98, 0xDA, 0x48, 0x36, 0x1C,
	0x55, 0xD3, 0x9A, 0x69, 0x16, 0x3F, 0xA8,
	0xFD, 0x24, 0xCF, 0x5F, 0x83, 0x65, 0x5D,
	0x23, 0xDC, 0xA3, 0xAD, 0x96, 0x1C, 0x62,
	0xF3, 0x56, 0x20, 0x85, 0x52, 0xBB, 0x9E,
	0xD5, 0x29, 0x07, 0x70, 0x96, 0x96, 0x6D,
	0x67, 0x0C, 0x35, 0x4E, 0x4A, 0xBC, 0x98,
	0x04, 0xF1, 0x74, 0x6C, 0x08, 0xCA, 0x18,
	0x21, 0x7C, 0x32, 0x90, 0x5E, 0x46, 0x2E,
	0x36, 0xCE, 0x3B, 0xE3, 0x9E, 0x77, 0x2C,
	0x18, 0x0E, 0x86, 0x03, 0x9B, 0x27, 0x83,
	0xA2, 0xEC, 0x07, 0xA2, 0x8F, 0xB5, 0xC5,
	0x5D, 0xF0, 0x6F, 0x4C, 0x52, 0xC9, 0xDE,
	0x2B, 0xCB, 0xF6, 0x95, 0x58, 0x17, 0x18,
	0x39, 0x95, 0x49, 0x7C, 0xEA, 0x95, 0x6A,
	0xE5, 0x15, 0xD2, 0x26, 0x18, 0x98, 0xFA,
	0x05, 0x10, 0x15, 0x72, 0x8E, 0x5A, 0x8A,
	0xAA, 0xC4, 0x2D, 0xAD, 0x33, 0x17, 0x0D,
	0x04, 0x50, 0x7A, 0x33, 0xA8, 0x55, 0x21,
	0xAB, 0xDF, 0x1C, 0xBA, 0x64, 0xEC, 0xFB,
	0x85, 0x04, 0x58, 0xDB, 0xEF, 0x0A, 0x8A,
	0xEA, 0x71, 0x57, 0x5D, 0x06, 0x0C, 0x7D,
	0xB3, 0x97, 0x0F, 0x85, 0xA6, 0xE1, 0xE4,
	0xC7, 0xAB, 0xF5, 0xAE, 0x8C, 0xDB, 0x09,
	0x33, 0xD7, 0x1E, 0x8C, 0x94, 0xE0, 0x4A,
	0x25, 0x61, 0x9D, 0xCE, 0xE3, 0xD2, 0x26,
	0x1A, 0xD2, 0xEE, 0x6B, 0xF1, 0x2F, 0xFA,
	0x06, 0xD9, 0x8A, 0x08, 0x64, 0xD8, 0x76,
	0x02, 0x73, 0x3E, 0xC8, 0x6A, 0x64, 0x52,
	0x1F, 0x2B, 0x18, 0x17, 0x7B, 0x20, 0x0C,
	0xBB, 0xE1, 0x17, 0x57, 0x7A, 0x61, 0x5D,
	0x6C, 0x77, 0x09, 0x88, 0xC0, 0xBA, 0xD9,
	0x46, 0xE2, 0x08, 0xE2, 0x4F, 0xA0, 0x74,
	0xE5, 0xAB, 0x31, 0x43, 0xDB, 0x5B, 0xFC,
	0xE0, 0xFD, 0x10, 0x8E, 0x4B, 0x82, 0xD1,
	0x20, 0xA9, 0x21, 0x08, 0x01, 0x1A, 0x72,
	0x3C, 0x12, 0xA7, 0x87, 0xE6, 0xD7, 0x88,
	0x71, 0x9A, 0x10, 0xBD, 0xBA, 0x5B, 0x26,
	0x99, 0xC3, 0x27, 0x18, 0x6A, 0xF4, 0xE2,
	0x3C, 0x1A, 0x94, 0x68, 0x34, 0xB6, 0x15,
	0x0B, 0xDA, 0x25, 0x83, 0xE9, 0xCA, 0x2A,
	0xD4, 0x4C, 0xE8, 0xDB, 0xBB, 0xC2, 0xDB,
	0x04, 0xDE, 0x8E, 0xF9, 0x2E, 0x8E, 0xFC,
	0x14, 0x1F, 0xBE, 0xCA, 0xA6, 0x28, 0x7C,
	0x59, 0x47, 0x4E, 0x6B, 0xC0, 0x5D, 0x99,
	0xB2, 0x96, 0x4F, 0xA0, 0x90, 0xC3, 0xA2,
	0x23, 0x3B, 0xA1, 0x86, 0x51, 0x5B, 0xE7,
	0xED, 0x1F, 0x61, 0x29, 0x70, 0xCE, 0xE2,
	0xD7, 0xAF, 0xB8, 0x1B, 0xDD, 0x76, 0x21,
	0x70, 0x48, 0x1C, 0xD0, 0x06, 0x91, 0x27,
	0xD5, 0xB0, 0x5A, 0xA9, 0x93, 0xB4, 0xEA,
	0x98, 0x8D, 0x8F, 0xDD, 0xC1, 0x86, 0xFF,
	0xB7, 0xDC, 0x90, 0xA6, 0xC0, 0x8F, 0x4D,
	0xF4, 0x35, 0xC9, 0x34, 0x06, 0x31, 0x99,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF
};

static const unsigned char modp_q_4096[] = {
	0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xE4, 0x87, 0xED, 0x51, 0x10, 0xB4,
	0x61, 0x1A, 0x62, 0x63, 0x31, 0x45, 0xC0,
	0x6E, 0x0E, 0x68, 0x94, 0x81, 0x27, 0x04,
	0x45, 0x33, 0xE6, 0x3A, 0x01, 0x05, 0xDF,
	0x53, 0x1D, 0x89, 0xCD, 0x91, 0x28, 0xA5,
	0x04, 0x3C, 0xC7, 0x1A, 0x02, 0x6E, 0xF7,
	0xCA, 0x8C, 0xD9, 0xE6, 0x9D, 0x21, 0x8D,
	0x98, 0x15, 0x85, 0x36, 0xF9, 0x2F, 0x8A,
	0x1B, 0xA7, 0xF0, 0x9A, 0xB6, 0xB6, 0xA8,
	0xE1, 0x22, 0xF2, 0x42, 0xDA, 0xBB, 0x31,
	0x2F, 0x3F, 0x63, 0x7A, 0x26, 0x21, 0x74,
	0xD3, 0x1B, 0xF6, 0xB5, 0x85, 0xFF, 0xAE,
	0x5B, 0x7A, 0x03, 0x5B, 0xF6, 0xF7, 0x1C,
	0x35, 0xFD, 0xAD, 0x44, 0xCF, 0xD2, 0xD7,
	0x4F, 0x92, 0x08, 0xBE, 0x25, 0x8F, 0xF3,
	0x24, 0x94, 0x33, 0x28, 0xF6, 0x72, 0x2D,
	0x9E, 0xE1, 0x00, 0x3E, 0x5C, 0x50, 0xB1,
	0xDF, 0x82, 0xCC, 0x6D, 0x24, 0x1B, 0x0E,
	0x2A, 0xE9, 0xCD, 0x34, 0x8B, 0x1F, 0xD4,
	0x7E, 0x92, 0x67, 0xAF, 0xC1, 0xB2, 0xAE,
	0x91, 0xEE, 0x51, 0xD6, 0xCB, 0x0E, 0x31,
	0x79, 0xAB, 0x10, 0x42, 0xA9, 0x5D, 0xCF,
	0x6A, 0x94, 0x83, 0xB8, 0x4B, 0x4B, 0x36,
	0xB3, 0x86, 0x1A, 0xA7, 0x25, 0x5E, 0x4C,
	0x02, 0x78, 0xBA, 0x36, 0x04, 0x65, 0x0C,
	0x10, 0xBE, 0x19, 0x48, 0x2F, 0x23, 0x17,
	0x1B, 0x67, 0x1D, 0xF1, 0xCF, 0x3B, 0x96,
	0x0C, 0x07, 0x43, 0x01, 0xCD, 0x93, 0xC1,
	0xD1, 0x76, 0x03, 0xD1, 0x47, 0xDA, 0xE2,
	0xAE, 0xF8, 0x37, 0xA6, 0x29, 0x64, 0xEF,
	0x15, 0xE5, 0xFB, 0x4A, 0xAC, 0x0B, 0x8C,
	0x1C, 0xCA, 0xA4, 0xBE, 0x75, 0x4A, 0xB5,
	0x72, 0x8A, 0xE9, 0x13, 0x0C, 0x4C, 0x7D,
	0x02, 0x88, 0x0A, 0xB9, 0x47, 0x2D, 0x45,
	0x55, 0x62, 0x16, 0xD6, 0x99, 0x8B, 0x86,
	0x82, 0x28, 0x3D, 0x19, 0xD4, 0x2A, 0x90,
	0xD5, 0xEF, 0x8E, 0x5D, 0x32, 0x76, 0x7D,
	0xC2, 0x82, 0x2C, 0x6D, 0xF7, 0x85, 0x45,
	0x75, 0x38, 0xAB, 0xAE, 0x83, 0x06, 0x3E,
	0xD9, 0xCB, 0x87, 0xC2, 0xD3, 0x70, 0xF2,
	0x63, 0xD5, 0xFA, 0xD7, 0x46, 0x6D, 0x84,
	0x99, 0xEB, 0x8F, 0x46, 0x4A, 0x70, 0x25,
	0x12, 0xB0, 0xCE, 0xE7, 0x71, 0xE9, 0x13,
	0x0D, 0x69, 0x77, 0x35, 0xF8, 0x97, 0xFD,
	0x03, 0x6C, 0xC5, 0x04, 0x32, 0x6C, 0x3B,
	0x01, 0x39, 0x9F, 0x64, 0x35, 0x32, 0x29,
	0x0F, 0x95, 0x8C, 0x0B, 0xBD, 0x90, 0x06,
	0x5D, 0xF0, 0x8B, 0xAB, 0xBD, 0x30, 0xAE,
	0xB6, 0x3B, 0x84, 0xC4, 0x60, 0x5D, 0x6C,
	0xA3, 0x71, 0x04, 0x71, 0x27, 0xD0, 0x3A,
	0x72, 0xD5, 0x98, 0xA1, 0xED, 0xAD, 0xFE,
	0x70, 0x7E, 0x88, 0x47, 0x25, 0xC1, 0x68,
	0x90, 0x54, 0x90, 0x84, 0x00, 0x8D, 0x39,
	0x1E, 0x09, 0x53, 0xC3, 0xF3, 0x6B, 0xC4,
	0x38, 0xCD, 0x08, 0x5E, 0xDD, 0x2D, 0x93,
	0x4C, 0xE1, 0x93, 0x8C, 0x35, 0x7A, 0x71,
	0x1E, 0x0D, 0x4A, 0x34, 0x1A, 0x5B, 0x0A,
	0x85, 0xED, 0x12, 0xC1, 0xF4, 0xE5, 0x15,
	0x6A, 0x26, 0x74, 0x6D, 0xDD, 0xE1, 0x6D,
	0x82, 0x6F, 0x47, 0x7C, 0x97, 0x47, 0x7E,
	0x0A, 0x0F, 0xDF, 0x65, 0x53, 0x14, 0x3E,
	0x2C, 0xA3, 0xA7, 0x35, 0xE0, 0x2E, 0xCC,
	0xD9, 0x4B, 0x27, 0xD0, 0x48, 0x61, 0xD1,
	0x11, 0x9D, 0xD0, 0xC3, 0x28, 0xAD, 0xF3,
	0xF6, 0x8F, 0xB0, 0x94, 0xB8, 0x67, 0x71,
	0x6B, 0xD7, 0xDC, 0x0D, 0xEE, 0xBB, 0x10,
	0xB8, 0x24, 0x0E, 0x68, 0x03, 0x48, 0x93,
	0xEA, 0xD8, 0x2D, 0x54, 0xC9, 0xDA, 0x75,
	0x4C, 0x46, 0xC7, 0xEE, 0xE0, 0xC3, 0x7F,
	0xDB, 0xEE, 0x48, 0x53, 0x60, 0x47, 0xA6,
	0xFA, 0x1A, 0xE4, 0x9A, 0x03, 0x18, 0xCC,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF
};

const gnutls_datum_t gnutls_modp_4096_group_prime = {
	(void *) modp_params_4096, sizeof(modp_params_4096)
};
const gnutls_datum_t gnutls_modp_4096_group_q = {
	(void *) modp_q_4096, sizeof(modp_q_4096)
};
const gnutls_datum_t gnutls_modp_4096_group_generator = {
	(void *) &modp_generator, sizeof(modp_generator)
};
const unsigned int gnutls_modp_4096_key_bits = 336;

static const unsigned char modp_params_6144[] = {
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xC9, 0x0F, 0xDA, 0xA2, 0x21, 0x68,
	0xC2, 0x34, 0xC4, 0xC6, 0x62, 0x8B, 0x80,
	0xDC, 0x1C, 0xD1, 0x29, 0x02, 0x4E, 0x08,
	0x8A, 0x67, 0xCC, 0x74, 0x02, 0x0B, 0xBE,
	0xA6, 0x3B, 0x13, 0x9B, 0x22, 0x51, 0x4A,
	0x08, 0x79, 0x8E, 0x34, 0x04, 0xDD, 0xEF,
	0x95, 0x19, 0xB3, 0xCD, 0x3A, 0x43, 0x1B,
	0x30, 0x2B, 0x0A, 0x6D, 0xF2, 0x5F, 0x14,
	0x37, 0x4F, 0xE1, 0x35, 0x6D, 0x6D, 0x51,
	0xC2, 0x45, 0xE4, 0x85, 0xB5, 0x76, 0x62,
	0x5E, 0x7E, 0xC6, 0xF4, 0x4C, 0x42, 0xE9,
	0xA6, 0x37, 0xED, 0x6B, 0x0B, 0xFF, 0x5C,
	0xB6, 0xF4, 0x06, 0xB7, 0xED, 0xEE, 0x38,
	0x6B, 0xFB, 0x5A, 0x89, 0x9F, 0xA5, 0xAE,
	0x9F, 0x24, 0x11, 0x7C, 0x4B, 0x1F, 0xE6,
	0x49, 0x28, 0x66, 0x51, 0xEC, 0xE4, 0x5B,
	0x3D, 0xC2, 0x00, 0x7C, 0xB8, 0xA1, 0x63,
	0xBF, 0x05, 0x98, 0xDA, 0x48, 0x36, 0x1C,
	0x55, 0xD3, 0x9A, 0x69, 0x16, 0x3F, 0xA8,
	0xFD, 0x24, 0xCF, 0x5F, 0x83, 0x65, 0x5D,
	0x23, 0xDC, 0xA3, 0xAD, 0x96, 0x1C, 0x62,
	0xF3, 0x56, 0x20, 0x85, 0x52, 0xBB, 0x9E,
	0xD5, 0x29, 0x07, 0x70, 0x96, 0x96, 0x6D,
	0x67, 0x0C, 0x35, 0x4E, 0x4A, 0xBC, 0x98,
	0x04, 0xF1, 0x74, 0x6C, 0x08, 0xCA, 0x18,
	0x21, 0x7C, 0x32, 0x90, 0x5E, 0x46, 0x2E,
	0x36, 0xCE, 0x3B, 0xE3, 0x9E, 0x77, 0x2C,
	0x18, 0x0E, 0x86, 0x03, 0x9B, 0x27, 0x83,
	0xA2, 0xEC, 0x07, 0xA2, 0x8F, 0xB5, 0xC5,
	0x5D, 0xF0, 0x6F, 0x4C, 0x52, 0xC9, 0xDE,
	0x2B, 0xCB, 0xF6, 0x95, 0x58, 0x17, 0x18,
	0x39, 0x95, 0x49, 0x7C, 0xEA, 0x95, 0x6A,
	0xE5, 0x15, 0xD2, 0x26, 0x18, 0x98, 0xFA,
	0x05, 0x10, 0x15, 0x72, 0x8E, 0x5A, 0x8A,
	0xAA, 0xC4, 0x2D, 0xAD, 0x33, 0x17, 0x0D,
	0x04, 0x50, 0x7A, 0x33, 0xA8, 0x55, 0x21,
	0xAB, 0xDF, 0x1C, 0xBA, 0x64, 0xEC, 0xFB,
	0x85, 0x04, 0x58, 0xDB, 0xEF, 0x0A, 0x8A,
	0xEA, 0x71, 0x57, 0x5D, 0x06, 0x0C, 0x7D,
	0xB3, 0x97, 0x0F, 0x85, 0xA6, 0xE1, 0xE4,
	0xC7, 0xAB, 0xF5, 0xAE, 0x8C, 0xDB, 0x09,
	0x33, 0xD7, 0x1E, 0x8C, 0x94, 0xE0, 0x4A,
	0x25, 0x61, 0x9D, 0xCE, 0xE3, 0xD2, 0x26,
	0x1A, 0xD2, 0xEE, 0x6B, 0xF1, 0x2F, 0xFA,
	0x06, 0xD9, 0x8A, 0x08, 0x64, 0xD8, 0x76,
	0x02, 0x73, 0x3E, 0xC8, 0x6A, 0x64, 0x52,
	0x1F, 0x2B, 0x18, 0x17, 0x7B, 0x20, 0x0C,
	0xBB, 0xE1, 0x17, 0x57, 0x7A, 0x61, 0x5D,
	0x6C, 0x77, 0x09, 0x88, 0xC0, 0xBA, 0xD9,
	0x46, 0xE2, 0x08, 0xE2, 0x4F, 0xA0, 0x74,
	0xE5, 0xAB, 0x31, 0x43, 0xDB, 0x5B, 0xFC,
	0xE0, 0xFD, 0x10, 0x8E, 0x4B, 0x82, 0xD1,
	0x20, 0xA9, 0x21, 0x08, 0x01, 0x1A, 0x72,
	0x3C, 0x12, 0xA7, 0x87, 0xE6, 0xD7, 0x88,
	0x71, 0x9A, 0x10, 0xBD, 0xBA, 0x5B, 0x26,
	0x99, 0xC3, 0x27, 0x18, 0x6A, 0xF4, 0xE2,
	0x3C, 0x1A, 0x94, 0x68, 0x34, 0xB6, 0x15,
	0x0B, 0xDA, 0x25, 0x83, 0xE9, 0xCA, 0x2A,
	0xD4, 0x4C, 0xE8, 0xDB, 0xBB, 0xC2, 0xDB,
	0x04, 0xDE, 0x8E, 0xF9, 0x2E, 0x8E, 0xFC,
	0x14, 0x1F, 0xBE, 0xCA, 0xA6, 0x28, 0x7C,
	0x59, 0x47, 0x4E, 0x6B, 0xC0, 0x5D, 0x99,
	0xB2, 0x96, 0x4F, 0xA0, 0x90, 0xC3, 0xA2,
	0x23, 0x3B, 0xA1, 0x86, 0x51, 0x5B, 0xE7,
	0xED, 0x1F, 0x61, 0x29, 0x70, 0xCE, 0xE2,
	0xD7, 0xAF, 0xB8, 0x1B, 0xDD, 0x76, 0x21,
	0x70, 0x48, 0x1C, 0xD0, 0x06, 0x91, 0x27,
	0xD5, 0xB0, 0x5A, 0xA9, 0x93, 0xB4, 0xEA,
	0x98, 0x8D, 0x8F, 0xDD, 0xC1, 0x86, 0xFF,
	0xB7, 0xDC, 0x90, 0xA6, 0xC0, 0x8F, 0x4D,
	0xF4, 0x35, 0xC9, 0x34, 0x02, 0x84, 0x92,
	0x36, 0xC3, 0xFA, 0xB4, 0xD2, 0x7C, 0x70,
	0x26, 0xC1, 0xD4, 0xDC, 0xB2, 0x60, 0x26,
	0x46, 0xDE, 0xC9, 0x75, 0x1E, 0x76, 0x3D,
	0xBA, 0x37, 0xBD, 0xF8, 0xFF, 0x94, 0x06,
	0xAD, 0x9E, 0x53, 0x0E, 0xE5, 0xDB, 0x38,
	0x2F, 0x41, 0x30, 0x01, 0xAE, 0xB0, 0x6A,
	0x53, 0xED, 0x90, 0x27, 0xD8, 0x31, 0x17,
	0x97, 0x27, 0xB0, 0x86, 0x5A, 0x89, 0x18,
	0xDA, 0x3E, 0xDB, 0xEB, 0xCF, 0x9B, 0x14,
	0xED, 0x44, 0xCE, 0x6C, 0xBA, 0xCE, 0xD4,
	0xBB, 0x1B, 0xDB, 0x7F, 0x14, 0x47, 0xE6,
	0xCC, 0x25, 0x4B, 0x33, 0x20, 0x51, 0x51,
	0x2B, 0xD7, 0xAF, 0x42, 0x6F, 0xB8, 0xF4,
	0x01, 0x37, 0x8C, 0xD2, 0xBF, 0x59, 0x83,
	0xCA, 0x01, 0xC6, 0x4B, 0x92, 0xEC, 0xF0,
	0x32, 0xEA, 0x15, 0xD1, 0x72, 0x1D, 0x03,
	0xF4, 0x82, 0xD7, 0xCE, 0x6E, 0x74, 0xFE,
	0xF6, 0xD5, 0x5E, 0x70, 0x2F, 0x46, 0x98,
	0x0C, 0x82, 0xB5, 0xA8, 0x40, 0x31, 0x90,
	0x0B, 0x1C, 0x9E, 0x59, 0xE7, 0xC9, 0x7F,
	0xBE, 0xC7, 0xE8, 0xF3, 0x23, 0xA9, 0x7A,
	0x7E, 0x36, 0xCC, 0x88, 0xBE, 0x0F, 0x1D,
	0x45, 0xB7, 0xFF, 0x58, 0x5A, 0xC5, 0x4B,
	0xD4, 0x07, 0xB2, 0x2B, 0x41, 0x54, 0xAA,
	0xCC, 0x8F, 0x6D, 0x7E, 0xBF, 0x48, 0xE1,
	0xD8, 0x14, 0xCC, 0x5E, 0xD2, 0x0F, 0x80,
	0x37, 0xE0, 0xA7, 0x97, 0x15, 0xEE, 0xF2,
	0x9B, 0xE3, 0x28, 0x06, 0xA1, 0xD5, 0x8B,
	0xB7, 0xC5, 0xDA, 0x76, 0xF5, 0x50, 0xAA,
	0x3D, 0x8A, 0x1F, 0xBF, 0xF0, 0xEB, 0x19,
	0xCC, 0xB1, 0xA3, 0x13, 0xD5, 0x5C, 0xDA,
	0x56, 0xC9, 0xEC, 0x2E, 0xF2, 0x96, 0x32,
	0x38, 0x7F, 0xE8, 0xD7, 0x6E, 0x3C, 0x04,
	0x68, 0x04, 0x3E, 0x8F, 0x66, 0x3F, 0x48,
	0x60, 0xEE, 0x12, 0xBF, 0x2D, 0x5B, 0x0B,
	0x74, 0x74, 0xD6, 0xE6, 0x94, 0xF9, 0x1E,
	0x6D, 0xCC, 0x40, 0x24, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

static const unsigned char modp_q_6144[] = {
	0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xE4, 0x87, 0xED, 0x51, 0x10, 0xB4,
	0x61, 0x1A, 0x62, 0x63, 0x31, 0x45, 0xC0,
	0x6E, 0x0E, 0x68, 0x94, 0x81, 0x27, 0x04,
	0x45, 0x33, 0xE6, 0x3A, 0x01, 0x05, 0xDF,
	0x53, 0x1D, 0x89, 0xCD, 0x91, 0x28, 0xA5,
	0x04, 0x3C, 0xC7, 0x1A, 0x02, 0x6E, 0xF7,
	0xCA, 0x8C, 0xD9, 0xE6, 0x9D, 0x21, 0x8D,
	0x98, 0x15, 0x85, 0x36, 0xF9, 0x2F, 0x8A,
	0x1B, 0xA7, 0xF0, 0x9A, 0xB6, 0xB6, 0xA8,
	0xE1, 0x22, 0xF2, 0x42, 0xDA, 0xBB, 0x31,
	0x2F, 0x3F, 0x63, 0x7A, 0x26, 0x21, 0x74,
	0xD3, 0x1B, 0xF6, 0xB5, 0x85, 0xFF, 0xAE,
	0x5B, 0x7A, 0x03, 0x5B, 0xF6, 0xF7, 0x1C,
	0x35, 0xFD, 0xAD, 0x44, 0xCF, 0xD2, 0xD7,
	0x4F, 0x92, 0x08, 0xBE, 0x25, 0x8F, 0xF3,
	0x24, 0x94, 0x33, 0x28, 0xF6, 0x72, 0x2D,
	0x9E, 0xE1, 0x00, 0x3E, 0x5C, 0x50, 0xB1,
	0xDF, 0x82, 0xCC, 0x6D, 0x24, 0x1B, 0x0E,
	0x2A, 0xE9, 0xCD, 0x34, 0x8B, 0x1F, 0xD4,
	0x7E, 0x92, 0x67, 0xAF, 0xC1, 0xB2, 0xAE,
	0x91, 0xEE, 0x51, 0xD6, 0xCB, 0x0E, 0x31,
	0x79, 0xAB, 0x10, 0x42, 0xA9, 0x5D, 0xCF,
	0x6A, 0x94, 0x83, 0xB8, 0x4B, 0x4B, 0x36,
	0xB3, 0x86, 0x1A, 0xA7, 0x25, 0x5E, 0x4C,
	0x02, 0x78, 0xBA, 0x36, 0x04, 0x65, 0x0C,
	0x10, 0xBE, 0x19, 0x48, 0x2F, 0x23, 0x17,
	0x1B, 0x67, 0x1D, 0xF1, 0xCF, 0x3B, 0x96,
	0x0C, 0x07, 0x43, 0x01, 0xCD, 0x93, 0xC1,
	0xD1, 0x76, 0x03, 0xD1, 0x47, 0xDA, 0xE2,
	0xAE, 0xF8, 0x37, 0xA6, 0x29, 0x64, 0xEF,
	0x15, 0xE5, 0xFB, 0x4A, 0xAC, 0x0B, 0x8C,
	0x1C, 0xCA, 0xA4, 0xBE, 0x75, 0x4A, 0xB5,
	0x72, 0x8A, 0xE9, 0x13, 0x0C, 0x4C, 0x7D,
	0x02, 0x88, 0x0A, 0xB9, 0x47, 0x2D, 0x45,
	0x55, 0x62, 0x16, 0xD6, 0x99, 0x8B, 0x86,
	0x82, 0x28, 0x3D, 0x19, 0xD4, 0x2A, 0x90,
	0xD5, 0xEF, 0x8E, 0x5D, 0x32, 0x76, 0x7D,
	0xC2, 0x82, 0x2C, 0x6D, 0xF7, 0x85, 0x45,
	0x75, 0x38, 0xAB, 0xAE, 0x83, 0x06, 0x3E,
	0xD9, 0xCB, 0x87, 0xC2, 0xD3, 0x70, 0xF2,
	0x63, 0xD5, 0xFA, 0xD7, 0x46, 0x6D, 0x84,
	0x99, 0xEB, 0x8F, 0x46, 0x4A, 0x70, 0x25,
	0x12, 0xB0, 0xCE, 0xE7, 0x71, 0xE9, 0x13,
	0x0D, 0x69, 0x77, 0x35, 0xF8, 0x97, 0xFD,
	0x03, 0x6C, 0xC5, 0x04, 0x32, 0x6C, 0x3B,
	0x01, 0x39, 0x9F, 0x64, 0x35, 0x32, 0x29,
	0x0F, 0x95, 0x8C, 0x0B, 0xBD, 0x90, 0x06,
	0x5D, 0xF0, 0x8B, 0xAB, 0xBD, 0x30, 0xAE,
	0xB6, 0x3B, 0x84, 0xC4, 0x60, 0x5D, 0x6C,
	0xA3, 0x71, 0x04, 0x71, 0x27, 0xD0, 0x3A,
	0x72, 0xD5, 0x98, 0xA1, 0xED, 0xAD, 0xFE,
	0x70, 0x7E, 0x88, 0x47, 0x25, 0xC1, 0x68,
	0x90, 0x54, 0x90, 0x84, 0x00, 0x8D, 0x39,
	0x1E, 0x09, 0x53, 0xC3, 0xF3, 0x6B, 0xC4,
	0x38, 0xCD, 0x08, 0x5E, 0xDD, 0x2D, 0x93,
	0x4C, 0xE1, 0x93, 0x8C, 0x35, 0x7A, 0x71,
	0x1E, 0x0D, 0x4A, 0x34, 0x1A, 0x5B, 0x0A,
	0x85, 0xED, 0x12, 0xC1, 0xF4, 0xE5, 0x15,
	0x6A, 0x26, 0x74, 0x6D, 0xDD, 0xE1, 0x6D,
	0x82, 0x6F, 0x47, 0x7C, 0x97, 0x47, 0x7E,
	0x0A, 0x0F, 0xDF, 0x65, 0x53, 0x14, 0x3E,
	0x2C, 0xA3, 0xA7, 0x35, 0xE0, 0x2E, 0xCC,
	0xD9, 0x4B, 0x27, 0xD0, 0x48, 0x61, 0xD1,
	0x11, 0x9D, 0xD0, 0xC3, 0x28, 0xAD, 0xF3,
	0xF6, 0x8F, 0xB0, 0x94, 0xB8, 0x67, 0x71,
	0x6B, 0xD7, 0xDC, 0x0D, 0xEE, 0xBB, 0x10,
	0xB8, 0x24, 0x0E, 0x68, 0x03, 0x48, 0x93,
	0xEA, 0xD8, 0x2D, 0x54, 0xC9, 0xDA, 0x75,
	0x4C, 0x46, 0xC7, 0xEE, 0xE0, 0xC3, 0x7F,
	0xDB, 0xEE, 0x48, 0x53, 0x60, 0x47, 0xA6,
	0xFA, 0x1A, 0xE4, 0x9A, 0x01, 0x42, 0x49,
	0x1B, 0x61, 0xFD, 0x5A, 0x69, 0x3E, 0x38,
	0x13, 0x60, 0xEA, 0x6E, 0x59, 0x30, 0x13,
	0x23, 0x6F, 0x64, 0xBA, 0x8F, 0x3B, 0x1E,
	0xDD, 0x1B, 0xDE, 0xFC, 0x7F, 0xCA, 0x03,
	0x56, 0xCF, 0x29, 0x87, 0x72, 0xED, 0x9C,
	0x17, 0xA0, 0x98, 0x00, 0xD7, 0x58, 0x35,
	0x29, 0xF6, 0xC8, 0x13, 0xEC, 0x18, 0x8B,
	0xCB, 0x93, 0xD8, 0x43, 0x2D, 0x44, 0x8C,
	0x6D, 0x1F, 0x6D, 0xF5, 0xE7, 0xCD, 0x8A,
	0x76, 0xA2, 0x67, 0x36, 0x5D, 0x67, 0x6A,
	0x5D, 0x8D, 0xED, 0xBF, 0x8A, 0x23, 0xF3,
	0x66, 0x12, 0xA5, 0x99, 0x90, 0x28, 0xA8,
	0x95, 0xEB, 0xD7, 0xA1, 0x37, 0xDC, 0x7A,
	0x00, 0x9B, 0xC6, 0x69, 0x5F, 0xAC, 0xC1,
	0xE5, 0x00, 0xE3, 0x25, 0xC9, 0x76, 0x78,
	0x19, 0x75, 0x0A, 0xE8, 0xB9, 0x0E, 0x81,
	0xFA, 0x41, 0x6B, 0xE7, 0x37, 0x3A, 0x7F,
	0x7B, 0x6A, 0xAF, 0x38, 0x17, 0xA3, 0x4C,
	0x06, 0x41, 0x5A, 0xD4, 0x20, 0x18, 0xC8,
	0x05, 0x8E, 0x4F, 0x2C, 0xF3, 0xE4, 0xBF,
	0xDF, 0x63, 0xF4, 0x79, 0x91, 0xD4, 0xBD,
	0x3F, 0x1B, 0x66, 0x44, 0x5F, 0x07, 0x8E,
	0xA2, 0xDB, 0xFF, 0xAC, 0x2D, 0x62, 0xA5,
	0xEA, 0x03, 0xD9, 0x15, 0xA0, 0xAA, 0x55,
	0x66, 0x47, 0xB6, 0xBF, 0x5F, 0xA4, 0x70,
	0xEC, 0x0A, 0x66, 0x2F, 0x69, 0x07, 0xC0,
	0x1B, 0xF0, 0x53, 0xCB, 0x8A, 0xF7, 0x79,
	0x4D, 0xF1, 0x94, 0x03, 0x50, 0xEA, 0xC5,
	0xDB, 0xE2, 0xED, 0x3B, 0x7A, 0xA8, 0x55,
	0x1E, 0xC5, 0x0F, 0xDF, 0xF8, 0x75, 0x8C,
	0xE6, 0x58, 0xD1, 0x89, 0xEA, 0xAE, 0x6D,
	0x2B, 0x64, 0xF6, 0x17, 0x79, 0x4B, 0x19,
	0x1C, 0x3F, 0xF4, 0x6B, 0xB7, 0x1E, 0x02,
	0x34, 0x02, 0x1F, 0x47, 0xB3, 0x1F, 0xA4,
	0x30, 0x77, 0x09, 0x5F, 0x96, 0xAD, 0x85,
	0xBA, 0x3A, 0x6B, 0x73, 0x4A, 0x7C, 0x8F,
	0x36, 0xE6, 0x20, 0x12, 0x7F, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

const gnutls_datum_t gnutls_modp_6144_group_prime = {
	(void *) modp_params_6144, sizeof(modp_params_6144)
};
const gnutls_datum_t gnutls_modp_6144_group_q = {
	(void *) modp_q_6144, sizeof(modp_q_6144)
};
const gnutls_datum_t gnutls_modp_6144_group_generator = {
	(void *) &modp_generator, sizeof(modp_generator)
};
const unsigned int gnutls_modp_6144_key_bits = 376;

static const unsigned char modp_params_8192[] = {
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xC9, 0x0F, 0xDA, 0xA2, 0x21, 0x68,
	0xC2, 0x34, 0xC4, 0xC6, 0x62, 0x8B, 0x80,
	0xDC, 0x1C, 0xD1, 0x29, 0x02, 0x4E, 0x08,
	0x8A, 0x67, 0xCC, 0x74, 0x02, 0x0B, 0xBE,
	0xA6, 0x3B, 0x13, 0x9B, 0x22, 0x51, 0x4A,
	0x08, 0x79, 0x8E, 0x34, 0x04, 0xDD, 0xEF,
	0x95, 0x19, 0xB3, 0xCD, 0x3A, 0x43, 0x1B,
	0x30, 0x2B, 0x0A, 0x6D, 0xF2, 0x5F, 0x14,
	0x37, 0x4F, 0xE1, 0x35, 0x6D, 0x6D, 0x51,
	0xC2, 0x45, 0xE4, 0x85, 0xB5, 0x76, 0x62,
	0x5E, 0x7E, 0xC6, 0xF4, 0x4C, 0x42, 0xE9,
	0xA6, 0x37, 0xED, 0x6B, 0x0B, 0xFF, 0x5C,
	0xB6, 0xF4, 0x06, 0xB7, 0xED, 0xEE, 0x38,
	0x6B, 0xFB, 0x5A, 0x89, 0x9F, 0xA5, 0xAE,
	0x9F, 0x24, 0x11, 0x7C, 0x4B, 0x1F, 0xE6,
	0x49, 0x28, 0x66, 0x51, 0xEC, 0xE4, 0x5B,
	0x3D, 0xC2, 0x00, 0x7C, 0xB8, 0xA1, 0x63,
	0xBF, 0x05, 0x98, 0xDA, 0x48, 0x36, 0x1C,
	0x55, 0xD3, 0x9A, 0x69, 0x16, 0x3F, 0xA8,
	0xFD, 0x24, 0xCF, 0x5F, 0x83, 0x65, 0x5D,
	0x23, 0xDC, 0xA3, 0xAD, 0x96, 0x1C, 0x62,
	0xF3, 0x56, 0x20, 0x85, 0x52, 0xBB, 0x9E,
	0xD5, 0x29, 0x07, 0x70, 0x96, 0x96, 0x6D,
	0x67, 0x0C, 0x35, 0x4E, 0x4A, 0xBC, 0x98,
	0x04, 0xF1, 0x74, 0x6C, 0x08, 0xCA, 0x18,
	0x21, 0x7C, 0x32, 0x90, 0x5E, 0x46, 0x2E,
	0x36, 0xCE, 0x3B, 0xE3, 0x9E, 0x77, 0x2C,
	0x18, 0x0E, 0x86, 0x03, 0x9B, 0x27, 0x83,
	0xA2, 0xEC, 0x07, 0xA2, 0x8F, 0xB5, 0xC5,
	0x5D, 0xF0, 0x6F, 0x4C, 0x52, 0xC9, 0xDE,
	0x2B, 0xCB, 0xF6, 0x95, 0x58, 0x17, 0x18,
	0x39, 0x95, 0x49, 0x7C, 0xEA, 0x95, 0x6A,
	0xE5, 0x15, 0xD2, 0x26, 0x18, 0x98, 0xFA,
	0x05, 0x10, 0x15, 0x72, 0x8E, 0x5A, 0x8A,
	0xAA, 0xC4, 0x2D, 0xAD, 0x33, 0x17, 0x0D,
	0x04, 0x50, 0x7A, 0x33, 0xA8, 0x55, 0x21,
	0xAB, 0xDF, 0x1C, 0xBA, 0x64, 0xEC, 0xFB,
	0x85, 0x04, 0x58, 0xDB, 0xEF, 0x0A, 0x8A,
	0xEA, 0x71, 0x57, 0x5D, 0x06, 0x0C, 0x7D,
	0xB3, 0x97, 0x0F, 0x85, 0xA6, 0xE1, 0xE4,
	0xC7, 0xAB, 0xF5, 0xAE, 0x8C, 0xDB, 0x09,
	0x33, 0xD7, 0x1E, 0x8C, 0x94, 0xE0, 0x4A,
	0x25, 0x61, 0x9D, 0xCE, 0xE3, 0xD2, 0x26,
	0x1A, 0xD2, 0xEE, 0x6B, 0xF1, 0x2F, 0xFA,
	0x06, 0xD9, 0x8A, 0x08, 0x64, 0xD8, 0x76,
	0x02, 0x73, 0x3E, 0xC8, 0x6A, 0x64, 0x52,
	0x1F, 0x2B, 0x18, 0x17, 0x7B, 0x20, 0x0C,
	0xBB, 0xE1, 0x17, 0x57, 0x7A, 0x61, 0x5D,
	0x6C, 0x77, 0x09, 0x88, 0xC0, 0xBA, 0xD9,
	0x46, 0xE2, 0x08, 0xE2, 0x4F, 0xA0, 0x74,
	0xE5, 0xAB, 0x31, 0x43, 0xDB, 0x5B, 0xFC,
	0xE0, 0xFD, 0x10, 0x8E, 0x4B, 0x82, 0xD1,
	0x20, 0xA9, 0x21, 0x08, 0x01, 0x1A, 0x72,
	0x3C, 0x12, 0xA7, 0x87, 0xE6, 0xD7, 0x88,
	0x71, 0x9A, 0x10, 0xBD, 0xBA, 0x5B, 0x26,
	0x99, 0xC3, 0x27, 0x18, 0x6A, 0xF4, 0xE2,
	0x3C, 0x1A, 0x94, 0x68, 0x34, 0xB6, 0x15,
	0x0B, 0xDA, 0x25, 0x83, 0xE9, 0xCA, 0x2A,
	0xD4, 0x4C, 0xE8, 0xDB, 0xBB, 0xC2, 0xDB,
	0x04, 0xDE, 0x8E, 0xF9, 0x2E, 0x8E, 0xFC,
	0x14, 0x1F, 0xBE, 0xCA, 0xA6, 0x28, 0x7C,
	0x59, 0x47, 0x4E, 0x6B, 0xC0, 0x5D, 0x99,
	0xB2, 0x96, 0x4F, 0xA0, 0x90, 0xC3, 0xA2,
	0x23, 0x3B, 0xA1, 0x86, 0x51, 0x5B, 0xE7,
	0xED, 0x1F, 0x61, 0x29, 0x70, 0xCE, 0xE2,
	0xD7, 0xAF, 0xB8, 0x1B, 0xDD, 0x76, 0x21,
	0x70, 0x48, 0x1C, 0xD0, 0x06, 0x91, 0x27,
	0xD5, 0xB0, 0x5A, 0xA9, 0x93, 0xB4, 0xEA,
	0x98, 0x8D, 0x8F, 0xDD, 0xC1, 0x86, 0xFF,
	0xB7, 0xDC, 0x90, 0xA6, 0xC0, 0x8F, 0x4D,
	0xF4, 0x35, 0xC9, 0x34, 0x02, 0x84, 0x92,
	0x36, 0xC3, 0xFA, 0xB4, 0xD2, 0x7C, 0x70,
	0x26, 0xC1, 0xD4, 0xDC, 0xB2, 0x60, 0x26,
	0x46, 0xDE, 0xC9, 0x75, 0x1E, 0x76, 0x3D,
	0xBA, 0x37, 0xBD, 0xF8, 0xFF, 0x94, 0x06,
	0xAD, 0x9E, 0x53, 0x0E, 0xE5, 0xDB, 0x38,
	0x2F, 0x41, 0x30, 0x01, 0xAE, 0xB0, 0x6A,
	0x53, 0xED, 0x90, 0x27, 0xD8, 0x31, 0x17,
	0x97, 0x27, 0xB0, 0x86, 0x5A, 0x89, 0x18,
	0xDA, 0x3E, 0xDB, 0xEB, 0xCF, 0x9B, 0x14,
	0xED, 0x44, 0xCE, 0x6C, 0xBA, 0xCE, 0xD4,
	0xBB, 0x1B, 0xDB, 0x7F, 0x14, 0x47, 0xE6,
	0xCC, 0x25, 0x4B, 0x33, 0x20, 0x51, 0x51,
	0x2B, 0xD7, 0xAF, 0x42, 0x6F, 0xB8, 0xF4,
	0x01, 0x37, 0x8C, 0xD2, 0xBF, 0x59, 0x83,
	0xCA, 0x01, 0xC6, 0x4B, 0x92, 0xEC, 0xF0,
	0x32, 0xEA, 0x15, 0xD1, 0x72, 0x1D, 0x03,
	0xF4, 0x82, 0xD7, 0xCE, 0x6E, 0x74, 0xFE,
	0xF6, 0xD5, 0x5E, 0x70, 0x2F, 0x46, 0x98,
	0x0C, 0x82, 0xB5, 0xA8, 0x40, 0x31, 0x90,
	0x0B, 0x1C, 0x9E, 0x59, 0xE7, 0xC9, 0x7F,
	0xBE, 0xC7, 0xE8, 0xF3, 0x23, 0xA9, 0x7A,
	0x7E, 0x36, 0xCC, 0x88, 0xBE, 0x0F, 0x1D,
	0x45, 0xB7, 0xFF, 0x58, 0x5A, 0xC5, 0x4B,
	0xD4, 0x07, 0xB2, 0x2B, 0x41, 0x54, 0xAA,
	0xCC, 0x8F, 0x6D, 0x7E, 0xBF, 0x48, 0xE1,
	0xD8, 0x14, 0xCC, 0x5E, 0xD2, 0x0F, 0x80,
	0x37, 0xE0, 0xA7, 0x97, 0x15, 0xEE, 0xF2,
	0x9B, 0xE3, 0x28, 0x06, 0xA1, 0xD5, 0x8B,
	0xB7, 0xC5, 0xDA, 0x76, 0xF5, 0x50, 0xAA,
	0x3D, 0x8A, 0x1F, 0xBF, 0xF0, 0xEB, 0x19,
	0xCC, 0xB1, 0xA3, 0x13, 0xD5, 0x5C, 0xDA,
	0x56, 0xC9, 0xEC, 0x2E, 0xF2, 0x96, 0x32,
	0x38, 0x7F, 0xE8, 0xD7, 0x6E, 0x3C, 0x04,
	0x68, 0x04, 0x3E, 0x8F, 0x66, 0x3F, 0x48,
	0x60, 0xEE, 0x12, 0xBF, 0x2D, 0x5B, 0x0B,
	0x74, 0x74, 0xD6, 0xE6, 0x94, 0xF9, 0x1E,
	0x6D, 0xBE, 0x11, 0x59, 0x74, 0xA3, 0x92,
	0x6F, 0x12, 0xFE, 0xE5, 0xE4, 0x38, 0x77,
	0x7C, 0xB6, 0xA9, 0x32, 0xDF, 0x8C, 0xD8,
	0xBE, 0xC4, 0xD0, 0x73, 0xB9, 0x31, 0xBA,
	0x3B, 0xC8, 0x32, 0xB6, 0x8D, 0x9D, 0xD3,
	0x00, 0x74, 0x1F, 0xA7, 0xBF, 0x8A, 0xFC,
	0x47, 0xED, 0x25, 0x76, 0xF6, 0x93, 0x6B,
	0xA4, 0x24, 0x66, 0x3A, 0xAB, 0x63, 0x9C,
	0x5A, 0xE4, 0xF5, 0x68, 0x34, 0x23, 0xB4,
	0x74, 0x2B, 0xF1, 0xC9, 0x78, 0x23, 0x8F,
	0x16, 0xCB, 0xE3, 0x9D, 0x65, 0x2D, 0xE3,
	0xFD, 0xB8, 0xBE, 0xFC, 0x84, 0x8A, 0xD9,
	0x22, 0x22, 0x2E, 0x04, 0xA4, 0x03, 0x7C,
	0x07, 0x13, 0xEB, 0x57, 0xA8, 0x1A, 0x23,
	0xF0, 0xC7, 0x34, 0x73, 0xFC, 0x64, 0x6C,
	0xEA, 0x30, 0x6B, 0x4B, 0xCB, 0xC8, 0x86,
	0x2F, 0x83, 0x85, 0xDD, 0xFA, 0x9D, 0x4B,
	0x7F, 0xA2, 0xC0, 0x87, 0xE8, 0x79, 0x68,
	0x33, 0x03, 0xED, 0x5B, 0xDD, 0x3A, 0x06,
	0x2B, 0x3C, 0xF5, 0xB3, 0xA2, 0x78, 0xA6,
	0x6D, 0x2A, 0x13, 0xF8, 0x3F, 0x44, 0xF8,
	0x2D, 0xDF, 0x31, 0x0E, 0xE0, 0x74, 0xAB,
	0x6A, 0x36, 0x45, 0x97, 0xE8, 0x99, 0xA0,
	0x25, 0x5D, 0xC1, 0x64, 0xF3, 0x1C, 0xC5,
	0x08, 0x46, 0x85, 0x1D, 0xF9, 0xAB, 0x48,
	0x19, 0x5D, 0xED, 0x7E, 0xA1, 0xB1, 0xD5,
	0x10, 0xBD, 0x7E, 0xE7, 0x4D, 0x73, 0xFA,
	0xF3, 0x6B, 0xC3, 0x1E, 0xCF, 0xA2, 0x68,
	0x35, 0x90, 0x46, 0xF4, 0xEB, 0x87, 0x9F,
	0x92, 0x40, 0x09, 0x43, 0x8B, 0x48, 0x1C,
	0x6C, 0xD7, 0x88, 0x9A, 0x00, 0x2E, 0xD5,
	0xEE, 0x38, 0x2B, 0xC9, 0x19, 0x0D, 0xA6,
	0xFC, 0x02, 0x6E, 0x47, 0x95, 0x58, 0xE4,
	0x47, 0x56, 0x77, 0xE9, 0xAA, 0x9E, 0x30,
	0x50, 0xE2, 0x76, 0x56, 0x94, 0xDF, 0xC8,
	0x1F, 0x56, 0xE8, 0x80, 0xB9, 0x6E, 0x71,
	0x60, 0xC9, 0x80, 0xDD, 0x98, 0xED, 0xD3,
	0xDF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF
};

static const unsigned char modp_q_8192[] = {
	0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xE4, 0x87, 0xED, 0x51, 0x10, 0xB4,
	0x61, 0x1A, 0x62, 0x63, 0x31, 0x45, 0xC0,
	0x6E, 0x0E, 0x68, 0x94, 0x81, 0x27, 0x04,
	0x45, 0x33, 0xE6, 0x3A, 0x01, 0x05, 0xDF,
	0x53, 0x1D, 0x89, 0xCD, 0x91, 0x28, 0xA5,
	0x04, 0x3C, 0xC7, 0x1A, 0x02, 0x6E, 0xF7,
	0xCA, 0x8C, 0xD9, 0xE6, 0x9D, 0x21, 0x8D,
	0x98, 0x15, 0x85, 0x36, 0xF9, 0x2F, 0x8A,
	0x1B, 0xA7, 0xF0, 0x9A, 0xB6, 0xB6, 0xA8,
	0xE1, 0x22, 0xF2, 0x42, 0xDA, 0xBB, 0x31,
	0x2F, 0x3F, 0x63, 0x7A, 0x26, 0x21, 0x74,
	0xD3, 0x1B, 0xF6, 0xB5, 0x85, 0xFF, 0xAE,
	0x5B, 0x7A, 0x03, 0x5B, 0xF6, 0xF7, 0x1C,
	0x35, 0xFD, 0xAD, 0x44, 0xCF, 0xD2, 0xD7,
	0x4F, 0x92, 0x08, 0xBE, 0x25, 0x8F, 0xF3,
	0x24, 0x94, 0x33, 0x28, 0xF6, 0x72, 0x2D,
	0x9E, 0xE1, 0x00, 0x3E, 0x5C, 0x50, 0xB1,
	0xDF, 0x82, 0xCC, 0x6D, 0x24, 0x1B, 0x0E,
	0x2A, 0xE9, 0xCD, 0x34, 0x8B, 0x1F, 0xD4,
	0x7E, 0x92, 0x67, 0xAF, 0xC1, 0xB2, 0xAE,
	0x91, 0xEE, 0x51, 0xD6, 0xCB, 0x0E, 0x31,
	0x79, 0xAB, 0x10, 0x42, 0xA9, 0x5D, 0xCF,
	0x6A, 0x94, 0x83, 0xB8, 0x4B, 0x4B, 0x36,
	0xB3, 0x86, 0x1A, 0xA7, 0x25, 0x5E, 0x4C,
	0x02, 0x78, 0xBA, 0x36, 0x04, 0x65, 0x0C,
	0x10, 0xBE, 0x19, 0x48, 0x2F, 0x23, 0x17,
	0x1B, 0x67, 0x1D, 0xF1, 0xCF, 0x3B, 0x96,
	0x0C, 0x07, 0x43, 0x01, 0xCD, 0x93, 0xC1,
	0xD1, 0x76, 0x03, 0xD1, 0x47, 0xDA, 0xE2,
	0xAE, 0xF8, 0x37, 0xA6, 0x29, 0x64, 0xEF,
	0x15, 0xE5, 0xFB, 0x4A, 0xAC, 0x0B, 0x8C,
	0x1C, 0xCA, 0xA4, 0xBE, 0x75, 0x4A, 0xB5,
	0x72, 0x8A, 0xE9, 0x13, 0x0C, 0x4C, 0x7D,
	0x02, 0x88, 0x0A, 0xB9, 0x47, 0x2D, 0x45,
	0x55, 0x62, 0x16, 0xD6, 0x99, 0x8B, 0x86,
	0x82, 0x28, 0x3D, 0x19, 0xD4, 0x2A, 0x90,
	0xD5, 0xEF, 0x8E, 0x5D, 0x32, 0x76, 0x7D,
	0xC2, 0x82, 0x2C, 0x6D, 0xF7, 0x85, 0x45,
	0x75, 0x38, 0xAB, 0xAE, 0x83, 0x06, 0x3E,
	0xD9, 0xCB, 0x87, 0xC2, 0xD3, 0x70, 0xF2,
	0x63, 0xD5, 0xFA, 0xD7, 0x46, 0x6D, 0x84,
	0x99, 0xEB, 0x8F, 0x46, 0x4A, 0x70, 0x25,
	0x12, 0xB0, 0xCE, 0xE7, 0x71, 0xE9, 0x13,
	0x0D, 0x69, 0x77, 0x35, 0xF8, 0x97, 0xFD,
	0x03, 0x6C, 0xC5, 0x04, 0x32, 0x6C, 0x3B,
	0x01, 0x39, 0x9F, 0x64, 0x35, 0x32, 0x29,
	0x0F, 0x95, 0x8C, 0x0B, 0xBD, 0x90, 0x06,
	0x5D, 0xF0, 0x8B, 0xAB, 0xBD, 0x30, 0xAE,
	0xB6, 0x3B, 0x84, 0xC4, 0x60, 0x5D, 0x6C,
	0xA3, 0x71, 0x04, 0x71, 0x27, 0xD0, 0x3A,
	0x72, 0xD5, 0x98, 0xA1, 0xED, 0xAD, 0xFE,
	0x70, 0x7E, 0x88, 0x47, 0x25, 0xC1, 0x68,
	0x90, 0x54, 0x90, 0x84, 0x00, 0x8D, 0x39,
	0x1E, 0x09, 0x53, 0xC3, 0xF3, 0x6B, 0xC4,
	0x38, 0xCD, 0x08, 0x5E, 0xDD, 0x2D, 0x93,
	0x4C, 0xE1, 0x93, 0x8C, 0x35, 0x7A, 0x71,
	0x1E, 0x0D, 0x4A, 0x34, 0x1A, 0x5B, 0x0A,
	0x85, 0xED, 0x12, 0xC1, 0xF4, 0xE5, 0x15,
	0x6A, 0x26, 0x74, 0x6D, 0xDD, 0xE1, 0x6D,
	0x82, 0x6F, 0x47, 0x7C, 0x97, 0x47, 0x7E,
	0x0A, 0x0F, 0xDF, 0x65, 0x53, 0x14, 0x3E,
	0x2C, 0xA3, 0xA7, 0x35, 0xE0, 0x2E, 0xCC,
	0xD9, 0x4B, 0x27, 0xD0, 0x48, 0x61, 0xD1,
	0x11, 0x9D, 0xD0, 0xC3, 0x28, 0xAD, 0xF3,
	0xF6, 0x8F, 0xB0, 0x94, 0xB8, 0x67, 0x71,
	0x6B, 0xD7, 0xDC, 0x0D, 0xEE, 0xBB, 0x10,
	0xB8, 0x24, 0x0E, 0x68, 0x03, 0x48, 0x93,
	0xEA, 0xD8, 0x2D, 0x54, 0xC9, 0xDA, 0x75,
	0x4C, 0x46, 0xC7, 0xEE, 0xE0, 0xC3, 0x7F,
	0xDB, 0xEE, 0x48, 0x53, 0x60, 0x47, 0xA6,
	0xFA, 0x1A, 0xE4, 0x9A, 0x01, 0x42, 0x49,
	0x1B, 0x61, 0xFD, 0x5A, 0x69, 0x3E, 0x38,
	0x13, 0x60, 0xEA, 0x6E, 0x59, 0x30, 0x13,
	0x23, 0x6F, 0x64, 0xBA, 0x8F, 0x3B, 0x1E,
	0xDD, 0x1B, 0xDE, 0xFC, 0x7F, 0xCA, 0x03,
	0x56, 0xCF, 0x29, 0x87, 0x72, 0xED, 0x9C,
	0x17, 0xA0, 0x98, 0x00, 0xD7, 0x58, 0x35,
	0x29, 0xF6, 0xC8, 0x13, 0xEC, 0x18, 0x8B,
	0xCB, 0x93, 0xD8, 0x43, 0x2D, 0x44, 0x8C,
	0x6D, 0x1F, 0x6D, 0xF5, 0xE7, 0xCD, 0x8A,
	0x76, 0xA2, 0x67, 0x36, 0x5D, 0x67, 0x6A,
	0x5D, 0x8D, 0xED, 0xBF, 0x8A, 0x23, 0xF3,
	0x66, 0x12, 0xA5, 0x99, 0x90, 0x28, 0xA8,
	0x95, 0xEB, 0xD7, 0xA1, 0x37, 0xDC, 0x7A,
	0x00, 0x9B, 0xC6, 0x69, 0x5F, 0xAC, 0xC1,
	0xE5, 0x00, 0xE3, 0x25, 0xC9, 0x76, 0x78,
	0x19, 0x75, 0x0A, 0xE8, 0xB9, 0x0E, 0x81,
	0xFA, 0x41, 0x6B, 0xE7, 0x37, 0x3A, 0x7F,
	0x7B, 0x6A, 0xAF, 0x38, 0x17, 0xA3, 0x4C,
	0x06, 0x41, 0x5A, 0xD4, 0x20, 0x18, 0xC8,
	0x05, 0x8E, 0x4F, 0x2C, 0xF3, 0xE4, 0xBF,
	0xDF, 0x63, 0xF4, 0x79, 0x91, 0xD4, 0xBD,
	0x3F, 0x1B, 0x66, 0x44, 0x5F, 0x07, 0x8E,
	0xA2, 0xDB, 0xFF, 0xAC, 0x2D, 0x62, 0xA5,
	0xEA, 0x03, 0xD9, 0x15, 0xA0, 0xAA, 0x55,
	0x66, 0x47, 0xB6, 0xBF, 0x5F, 0xA4, 0x70,
	0xEC, 0x0A, 0x66, 0x2F, 0x69, 0x07, 0xC0,
	0x1B, 0xF0, 0x53, 0xCB, 0x8A, 0xF7, 0x79,
	0x4D, 0xF1, 0x94, 0x03, 0x50, 0xEA, 0xC5,
	0xDB, 0xE2, 0xED, 0x3B, 0x7A, 0xA8, 0x55,
	0x1E, 0xC5, 0x0F, 0xDF, 0xF8, 0x75, 0x8C,
	0xE6, 0x58, 0xD1, 0x89, 0xEA, 0xAE, 0x6D,
	0x2B, 0x64, 0xF6, 0x17, 0x79, 0x4B, 0x19,
	0x1C, 0x3F, 0xF4, 0x6B, 0xB7, 0x1E, 0x02,
	0x34, 0x02, 0x1F, 0x47, 0xB3, 0x1F, 0xA4,
	0x30, 0x77, 0x09, 0x5F, 0x96, 0xAD, 0x85,
	0xBA, 0x3A, 0x6B, 0x73, 0x4A, 0x7C, 0x8F,
	0x36, 0xDF, 0x08, 0xAC, 0xBA, 0x51, 0xC9,
	0x37, 0x89, 0x7F, 0x72, 0xF2, 0x1C, 0x3B,
	0xBE, 0x5B, 0x54, 0x99, 0x6F, 0xC6, 0x6C,
	0x5F, 0x62, 0x68, 0x39, 0xDC, 0x98, 0xDD,
	0x1D, 0xE4, 0x19, 0x5B, 0x46, 0xCE, 0xE9,
	0x80, 0x3A, 0x0F, 0xD3, 0xDF, 0xC5, 0x7E,
	0x23, 0xF6, 0x92, 0xBB, 0x7B, 0x49, 0xB5,
	0xD2, 0x12, 0x33, 0x1D, 0x55, 0xB1, 0xCE,
	0x2D, 0x72, 0x7A, 0xB4, 0x1A, 0x11, 0xDA,
	0x3A, 0x15, 0xF8, 0xE4, 0xBC, 0x11, 0xC7,
	0x8B, 0x65, 0xF1, 0xCE, 0xB2, 0x96, 0xF1,
	0xFE, 0xDC, 0x5F, 0x7E, 0x42, 0x45, 0x6C,
	0x91, 0x11, 0x17, 0x02, 0x52, 0x01, 0xBE,
	0x03, 0x89, 0xF5, 0xAB, 0xD4, 0x0D, 0x11,
	0xF8, 0x63, 0x9A, 0x39, 0xFE, 0x32, 0x36,
	0x75, 0x18, 0x35, 0xA5, 0xE5, 0xE4, 0x43,
	0x17, 0xC1, 0xC2, 0xEE, 0xFD, 0x4E, 0xA5,
	0xBF, 0xD1, 0x60, 0x43, 0xF4, 0x3C, 0xB4,
	0x19, 0x81, 0xF6, 0xAD, 0xEE, 0x9D, 0x03,
	0x15, 0x9E, 0x7A, 0xD9, 0xD1, 0x3C, 0x53,
	0x36, 0x95, 0x09, 0xFC, 0x1F, 0xA2, 0x7C,
	0x16, 0xEF, 0x98, 0x87, 0x70, 0x3A, 0x55,
	0xB5, 0x1B, 0x22, 0xCB, 0xF4, 0x4C, 0xD0,
	0x12, 0xAE, 0xE0, 0xB2, 0x79, 0x8E, 0x62,
	0x84, 0x23, 0x42, 0x8E, 0xFC, 0xD5, 0xA4,
	0x0C, 0xAE, 0xF6, 0xBF, 0x50, 0xD8, 0xEA,
	0x88, 0x5E, 0xBF, 0x73, 0xA6, 0xB9, 0xFD,
	0x79, 0xB5, 0xE1, 0x8F, 0x67, 0xD1, 0x34,
	0x1A, 0xC8, 0x23, 0x7A, 0x75, 0xC3, 0xCF,
	0xC9, 0x20, 0x04, 0xA1, 0xC5, 0xA4, 0x0E,
	0x36, 0x6B, 0xC4, 0x4D, 0x00, 0x17, 0x6A,
	0xF7, 0x1C, 0x15, 0xE4, 0x8C, 0x86, 0xD3,
	0x7E, 0x01, 0x37, 0x23, 0xCA, 0xAC, 0x72,
	0x23, 0xAB, 0x3B, 0xF4, 0xD5, 0x4F, 0x18,
	0x28, 0x71, 0x3B, 0x2B, 0x4A, 0x6F, 0xE4,
	0x0F, 0xAB, 0x74, 0x40, 0x5C, 0xB7, 0x38,
	0xB0, 0x64, 0xC0, 0x6E, 0xCC, 0x76, 0xE9,
	0xEF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF
};

const gnutls_datum_t gnutls_modp_8192_group_prime = {
	(void *) modp_params_8192, sizeof(modp_params_8192)
};
const gnutls_datum_t gnutls_modp_8192_group_q = {
	(void *) modp_q_8192, sizeof(modp_q_8192)
};
const gnutls_datum_t gnutls_modp_8192_group_generator = {
	(void *) &modp_generator, sizeof(modp_generator)
};
const unsigned int gnutls_modp_8192_key_bits = 512;

unsigned
_gnutls_dh_prime_match_fips_approved(const uint8_t *prime,
				     size_t prime_size,
				     const uint8_t *generator,
				     size_t generator_size,
				     uint8_t **q,
				     size_t *q_size)
{
	static const struct {
		const gnutls_datum_t *prime;
		const gnutls_datum_t *generator;
		const gnutls_datum_t *q;
	} primes[] = {
		{ &gnutls_ffdhe_8192_group_prime, &gnutls_ffdhe_8192_group_generator, &gnutls_ffdhe_8192_group_q },
		{ &gnutls_ffdhe_6144_group_prime, &gnutls_ffdhe_6144_group_generator, &gnutls_ffdhe_6144_group_q },
		{ &gnutls_ffdhe_4096_group_prime, &gnutls_ffdhe_4096_group_generator, &gnutls_ffdhe_4096_group_q },
		{ &gnutls_ffdhe_3072_group_prime, &gnutls_ffdhe_3072_group_generator, &gnutls_ffdhe_3072_group_q },
		{ &gnutls_ffdhe_2048_group_prime, &gnutls_ffdhe_2048_group_generator, &gnutls_ffdhe_2048_group_q },
		{ &gnutls_modp_8192_group_prime, &gnutls_modp_8192_group_generator, &gnutls_modp_8192_group_q },
		{ &gnutls_modp_6144_group_prime, &gnutls_modp_6144_group_generator, &gnutls_modp_6144_group_q },
		{ &gnutls_modp_4096_group_prime, &gnutls_modp_4096_group_generator, &gnutls_modp_4096_group_q },
		{ &gnutls_modp_3072_group_prime, &gnutls_modp_3072_group_generator, &gnutls_modp_3072_group_q },
		{ &gnutls_modp_2048_group_prime, &gnutls_modp_2048_group_generator, &gnutls_modp_2048_group_q },
	};
	size_t i;

	for (i = 0; i < sizeof(primes) / sizeof(primes[0]); i++) {
		if (primes[i].prime->size == prime_size &&
		    memcmp(primes[i].prime->data, prime, primes[i].prime->size) == 0 &&
		    primes[i].generator->size == generator_size &&
		    memcmp(primes[i].generator->data, generator, primes[i].generator->size) == 0) {
			if (q) {
				*q = primes[i].q->data;
				*q_size = primes[i].q->size;
			}
			return 1;
		}
	}

	return 0;
}

#endif
