# Licensed to the Apache Software Foundation (ASF) under one or more
# contributor license agreements.  See the NOTICE file distributed with
# this work for additional information regarding copyright ownership.
# The ASF licenses this file to You under the Apache License, Version 2.0
# (the "License"); you may not use this file except in compliance with
# the License.  You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

use strict;
use warnings;

package Clownfish::Binding::Core::File;
use Clownfish::Util qw( verify_args );

my %write_h_PARAMS = (
    file   => undef,
    dest   => undef,
    header => undef,
    footer => undef,
);

sub write_h {
    my ( undef, %args ) = @_;
    verify_args( \%write_h_PARAMS, %args ) or confess $@;
    _write_h( @args{qw( file dest header footer )} );
}

1;

__END__

__POD__

=head1 NAME

Clownfish::Binding::Core::File - Generate core C code for a Clownfish file.

=head1 DESCRIPTION

This module is the companion to Clownfish::File, generating the C code
needed to implement the file's specification.

There is a one-to-one mapping between Clownfish header files and autogenerated
.h and .c files.  If Foo.cfh includes both Foo and Foo::FooJr, then it is
necessary to pound-include "Foo.h" in order to get FooJr's interface -- not
"Foo/FooJr.h", which won't exist.

=head1 CLASS METHODS

=head2 write_h

    Clownfish::Binding::Core::File->write_c(
        file   => $file,                            # required
        dest   => '/path/to/autogen_dir',           # required
        header => "/* Autogenerated file. */\n",    # required
        footer => $copyfoot,                        # required
    );

Generate a C header file containing all class declarations and literal C
blocks.

=over

=item * B<file> - A L<Clownfish::File>.

=item * B<dest> - The directory under which autogenerated files are being
written.

=item * B<header> - Text which will be prepended to each generated C file --
typically, an "autogenerated file" warning.

=item * B<footer> - Text to be appended to the end of each generated C file --
typically copyright information.

=back 

=cut
