/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * Copyright by the Board of Trustees of the University of Illinois.         *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the COPYING file, which can be found at the root of the source code       *
 * distribution tree, or in https://support.hdfgroup.org/ftp/HDF5/releases.  *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*-------------------------------------------------------------------------
 *
 * Created:     H5Ocache_image.c
 *              June 21, 2015
 *              John Mainzer
 *
 * Purpose:     A message indicating that a metadata cache image block
 *		of the indicated length exists at the specified offset
 *		in the HDF5 file.
 *
 * 		The mdci_msg only appears in the superblock extension.
 *
 *-------------------------------------------------------------------------
 */

#include "H5Omodule.h"          /* This source code file is part of the H5O module */


#include "H5private.h"          /* Generic Functions                     */
#include "H5Eprivate.h"         /* Error handling                        */
#include "H5Fprivate.h"		/* Files				*/
#include "H5FLprivate.h"        /* Free Lists                            */
#include "H5Opkg.h"             /* Object headers                        */
#include "H5MFprivate.h"        /* File space management                 */

/* Callbacks for message class */
static void *H5O__mdci_decode(H5F_t *f, hid_t dxpl_id, H5O_t *open_oh,
    unsigned mesg_flags, unsigned *ioflags, size_t p_size, const uint8_t *p);
static herr_t H5O__mdci_encode(H5F_t *f, hbool_t disable_shared, 
    uint8_t *p, const void *_mesg);
static void *H5O__mdci_copy(const void *_mesg, void *_dest);
static size_t H5O__mdci_size(const H5F_t *f, hbool_t disable_shared, 
    const void *_mesg);
static herr_t H5O__mdci_free(void *mesg);
static herr_t H5O__mdci_delete(H5F_t *f, hid_t dxpl_id, H5O_t *open_oh,
    void *_mesg);
static herr_t H5O__mdci_debug(H5F_t *f, hid_t dxpl_id, const void *_mesg,
    FILE *stream, int indent, int fwidth);

/* This message derives from H5O message class */
const H5O_msg_class_t H5O_MSG_MDCI[1] = {{
    H5O_MDCI_MSG_ID,            /* message id number              */
    "mdci",                     /* message name for debugging     */
    sizeof(H5O_mdci_t),         /* native message size            */
    0,                          /* messages are sharable?         */
    H5O__mdci_decode,           /* decode message                 */
    H5O__mdci_encode,           /* encode message                 */
    H5O__mdci_copy,             /* copy method                    */
    H5O__mdci_size,             /* size of mdc image message      */
    NULL,                       /* reset method                   */
    H5O__mdci_free,             /* free method                    */
    H5O__mdci_delete,		/* file delete method             */
    NULL,                       /* link method                    */
    NULL,                       /* set share method               */
    NULL,                       /* can share method               */
    NULL,                       /* pre copy native value to file  */
    NULL,                       /* copy native value to file      */
    NULL,                       /* post copy native value to file */
    NULL,                       /* get creation index             */
    NULL,                       /* set creation index             */
    H5O__mdci_debug             /* debugging                      */
}};

/* Only one version of the metadata cache image message at present */
#define H5O_MDCI_VERSION_0 	0

/* Declare the free list for H5O_mdci_t's */
H5FL_DEFINE(H5O_mdci_t);



/*-------------------------------------------------------------------------
 * Function:    H5O__mdci_decode
 *
 * Purpose:     Decode a metadata cache image  message and return a
 * 		pointer to a newly allocated H5O_mdci_t struct.
 *
 * Return:      Success:        Ptr to new message in native struct.
 *              Failure:        NULL
 *
 * Programmer:  John Mainzer
 *              6/22/15
 *
 *-------------------------------------------------------------------------
 */
static void *
H5O__mdci_decode(H5F_t *f, hid_t H5_ATTR_UNUSED dxpl_id, 
    H5O_t H5_ATTR_UNUSED *open_oh, unsigned H5_ATTR_UNUSED mesg_flags, 
    unsigned H5_ATTR_UNUSED *ioflags, size_t H5_ATTR_UNUSED p_size, const uint8_t *p)
{
    H5O_mdci_t      *mesg;                  /* Native message        */
    void            *ret_value = NULL;      /* Return value          */

    FUNC_ENTER_STATIC

    /* Sanity check */
    HDassert(f);
    HDassert(p);

    /* Version of message */
    if(*p++ != H5O_MDCI_VERSION_0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "bad version number for message")

    /* Allocate space for message */
    if(NULL == (mesg = (H5O_mdci_t *)H5FL_MALLOC(H5O_mdci_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for metadata cache image message")

    /* Decode */
    H5F_addr_decode(f, &p, &(mesg->addr));
    H5F_DECODE_LENGTH(f, p, mesg->size);

    /* Set return value */
    ret_value = (void *)mesg;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} /* end H5O__mdci_decode() */


/*-------------------------------------------------------------------------
 * Function:    H5O__mdci_encode
 *
 * Purpose:     Encode metadata cache image message
 *
 * Return:      Non-negative on success/Negative on failure
 *
 * Programmer:  John Mainzer
 *              6/22/15
 *
 *-------------------------------------------------------------------------
 */
static herr_t
H5O__mdci_encode(H5F_t *f, hbool_t H5_ATTR_UNUSED disable_shared, 
    uint8_t *p, const void *_mesg)
{
    const H5O_mdci_t *mesg = (const H5O_mdci_t *)_mesg;

    FUNC_ENTER_STATIC_NOERR

    /* Sanity check */
    HDassert(f);
    HDassert(p);
    HDassert(mesg);

    /* encode */
    *p++ = H5O_MDCI_VERSION_0;
    H5F_addr_encode(f, &p, mesg->addr);
    H5F_ENCODE_LENGTH(f, p, mesg->size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} /* end H5O__mdci_encode() */


/*-------------------------------------------------------------------------
 * Function:    H5O__mdci_copy
 *
 * Purpose:     Copies a message from _MESG to _DEST, allocating _DEST if
 *              necessary.
 *
 * Return:      Success:        Ptr to _DEST
 *              Failure:        NULL
 *
 * Programmer:  John Mainzer
 *              6/22/15
 *
 *-------------------------------------------------------------------------
 */
static void *
H5O__mdci_copy(const void *_mesg, void *_dest)
{
    const H5O_mdci_t   *mesg = (const H5O_mdci_t *)_mesg;
    H5O_mdci_t         *dest = (H5O_mdci_t *) _dest;
    void               *ret_value = NULL;   /* Return value */

    FUNC_ENTER_STATIC

    /* check args */
    HDassert(mesg);
    if(!dest && NULL == (dest = H5FL_MALLOC(H5O_mdci_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed")

    /* copy */
    *dest = *mesg;

    /* Set return value */
    ret_value = dest;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} /* end H5O_mdci__copy() */


/*-------------------------------------------------------------------------
 * Function:    H5O__mdci_size
 *
 * Purpose:     Returns the size of the raw message in bytes not counting
 *              the message type or size fields, but only the data fields.
 *              This function doesn't take into account alignment.
 *
 * Return:      Success:        Message data size in bytes without alignment.
 *
 *              Failure:        zero
 *
 * Programmer:  John Mainzer
 *              6/22/15
 *
 *-------------------------------------------------------------------------
 */
static size_t
H5O__mdci_size(const H5F_t *f, hbool_t H5_ATTR_UNUSED disable_shared, 
    const void H5_ATTR_UNUSED *_mesg)
{
    size_t ret_value = 0;       /* Return value */

    FUNC_ENTER_STATIC_NOERR

    /* Set return value */
    ret_value = (size_t)( 1 +                   /* Version number           */ 
                          H5F_SIZEOF_ADDR(f) +  /* addr of metadata cache   */
                                                /* image block              */
                          H5F_SIZEOF_SIZE(f) ); /* length of metadata cache */
                                                /* image block              */

    FUNC_LEAVE_NOAPI(ret_value)
} /* end H5O__mdci_size() */


/*-------------------------------------------------------------------------
 * Function:    H5O__mdci_free
 *
 * Purpose:     Free the message
 *
 * Return:      Non-negative on success/Negative on failure
 *
 * Programmer:  John Mainzer
 *              6/22/15
 *
 *-------------------------------------------------------------------------
 */
static herr_t
H5O__mdci_free(void *mesg)
{
    FUNC_ENTER_STATIC_NOERR

    HDassert(mesg);

    mesg = H5FL_FREE(H5O_mdci_t, mesg);

    FUNC_LEAVE_NOAPI(SUCCEED)
} /* end H5O__mdci_free() */


/*-------------------------------------------------------------------------
 * Function:    H5O__mdci_delete
 *
 * Purpose:     Free file space referenced by message
 *
 * Return:      Non-negative on success/Negative on failure
 *
 * Programmer:  Quincey Koziol
 *              Wednesday, March 19, 2003
 *
 *-------------------------------------------------------------------------
 */
static herr_t
H5O__mdci_delete(H5F_t *f, hid_t dxpl_id, H5O_t *open_oh, void *_mesg)
{
    H5O_mdci_t *mesg = (H5O_mdci_t *)_mesg;
    herr_t ret_value = SUCCEED;   /* Return value */

    FUNC_ENTER_STATIC

    /* check args */
    HDassert(f);
    HDassert(open_oh);
    HDassert(mesg);

    /* Free file space for cache image */
    if(H5F_addr_defined(mesg->addr)) {
        /* The space for the cache image block was allocated directly
         * from the VFD layer at the end of file.  As this was the
         * last file space allocation before shutdown, the cache image
         * should still be the last item in the file.
         *
         * If the hack to work around the self referential free space
         * manager issue is in use, file space for the non-empty self
         * referential free space managers was also allocated from VFD
         * layer at the end of file.  Since these allocations directly
         * preceeded the cache image allocation they should be directly
         * adjacent to the cache image block at the end of file.
         *
         * In this case, just call H5MF_tidy_self_referential_fsm_hack().
         *
         * That routine will float the self referential free space
         * managers, and reduce the eoa to its value just prior to
         * allocation of space for same.  Since the cache image appears
         * just after the self referential free space managers, this
         * will release the file space for the cache image as well.
         *
         * Note that in this case, there must not have been any file
         * space allocations / deallocations prior to the free of the
         * cache image.  Verify this to the extent possible.
         *
         * If the hack to work around the persistant self referential
         * free space manager issue is NOT in use, just call H5MF_xfree()
         * to release the cache iamge.  In principle, we should be able
         * to just reduce the EOA to the base address of the cache
         * image block, as there shouldn't be any file space allocation
         * before the first metadata cache access.  However, given
         * time constraints, I don't want to go there now.
         */
        if(H5F_FIRST_ALLOC_DEALLOC(f)) {
            HDassert(HADDR_UNDEF != H5F_EOA_PRE_FSM_FSALLOC(f));
            HDassert(H5F_addr_ge(mesg->addr, H5F_EOA_PRE_FSM_FSALLOC(f)));
            if(H5MF_tidy_self_referential_fsm_hack(f, dxpl_id) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTFREE, FAIL, "tidy of self referential fsm hack failed")
        } /* end if */
        else {
            if(H5MF_xfree(f, H5FD_MEM_SUPER, dxpl_id, mesg->addr, mesg->size) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTFREE, FAIL, "unable to free file space for cache image block")
        } /* end else */
    } /* end if */

done:
    FUNC_LEAVE_NOAPI(ret_value)
} /* end H5O__mdci_delete() */


/*-------------------------------------------------------------------------
 * Function:    H5O__mdci_debug
 *
 * Purpose:     Prints debugging info.
 *
 * Return:      Non-negative on success/Negative on failure
 *
 * Programmer:  John Mainzer
 *              6/22/15
 *
 *-------------------------------------------------------------------------
 */
static herr_t
H5O__mdci_debug(H5F_t H5_ATTR_UNUSED *f, hid_t H5_ATTR_UNUSED dxpl_id, 
    const void *_mesg, FILE * stream, int indent, int fwidth)
{
    const H5O_mdci_t   *mdci = (const H5O_mdci_t *) _mesg;

    FUNC_ENTER_STATIC_NOERR

    /* check args */
    HDassert(f);
    HDassert(mdci);
    HDassert(stream);
    HDassert(indent >= 0);
    HDassert(fwidth >= 0);

    HDfprintf(stream, "%*s%-*s %a\n", indent, "", fwidth,
              "Metadata Cache Image Block address:", mdci->addr);

    HDfprintf(stream, "%*s%-*s %Hu\n", indent, "", fwidth,
              "Metadata Cache Image Block size in bytes:", mdci->size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} /* end H5O__mdci_debug() */

