/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/* vim: set ts=4 et sw=4 tw=80: */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/. */

#include <string.h>

#include "nss.h"
#include "pk11pub.h"
#include "prenv.h"
#include "prerror.h"
#include "secmod.h"

#include "gtest/gtest.h"
#include "nss_scoped_ptrs.h"
#include "util.h"

namespace nss_test {

// These test certificates were generated using pycert/pykey from
// mozilla-central (https://hg.mozilla.org/mozilla-central/file/ ...
// 9968319230a74eb8c1953444a0e6973c7500a9f8/security/manager/ssl/ ...
// tests/unit/pycert.py).

// issuer:test cert
// subject:test cert
// issuerKey:secp256r1
// subjectKey:secp256r1
// serialNumber:1
std::vector<uint8_t> kTestCert1DER = {
    0x30, 0x82, 0x01, 0x1D, 0x30, 0x81, 0xC2, 0xA0, 0x03, 0x02, 0x01, 0x02,
    0x02, 0x01, 0x01, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7,
    0x0D, 0x01, 0x01, 0x0B, 0x05, 0x00, 0x30, 0x14, 0x31, 0x12, 0x30, 0x10,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x09, 0x74, 0x65, 0x73, 0x74, 0x20,
    0x63, 0x65, 0x72, 0x74, 0x30, 0x22, 0x18, 0x0F, 0x32, 0x30, 0x31, 0x37,
    0x31, 0x31, 0x32, 0x37, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5A, 0x18,
    0x0F, 0x32, 0x30, 0x32, 0x30, 0x30, 0x32, 0x30, 0x35, 0x30, 0x30, 0x30,
    0x30, 0x30, 0x30, 0x5A, 0x30, 0x14, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03,
    0x55, 0x04, 0x03, 0x0C, 0x09, 0x74, 0x65, 0x73, 0x74, 0x20, 0x63, 0x65,
    0x72, 0x74, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE,
    0x3D, 0x02, 0x01, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x03, 0x01,
    0x07, 0x03, 0x42, 0x00, 0x04, 0x4F, 0xBF, 0xBB, 0xBB, 0x61, 0xE0, 0xF8,
    0xF9, 0xB1, 0xA6, 0x0A, 0x59, 0xAC, 0x87, 0x04, 0xE2, 0xEC, 0x05, 0x0B,
    0x42, 0x3E, 0x3C, 0xF7, 0x2E, 0x92, 0x3F, 0x2C, 0x4F, 0x79, 0x4B, 0x45,
    0x5C, 0x2A, 0x69, 0xD2, 0x33, 0x45, 0x6C, 0x36, 0xC4, 0x11, 0x9D, 0x07,
    0x06, 0xE0, 0x0E, 0xED, 0xC8, 0xD1, 0x93, 0x90, 0xD7, 0x99, 0x1B, 0x7B,
    0x2D, 0x07, 0xA3, 0x04, 0xEA, 0xA0, 0x4A, 0xA6, 0xC0, 0x30, 0x0D, 0x06,
    0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x0B, 0x05, 0x00,
    0x03, 0x47, 0x00, 0x30, 0x44, 0x02, 0x20, 0x5C, 0x75, 0x51, 0x9F, 0x13,
    0x11, 0x50, 0xCD, 0x5D, 0x8A, 0xDE, 0x20, 0xA3, 0xBC, 0x06, 0x30, 0x91,
    0xFF, 0xB2, 0x73, 0x75, 0x5F, 0x31, 0x64, 0xEC, 0xFD, 0xCB, 0x42, 0x80,
    0x0A, 0x70, 0xE6, 0x02, 0x20, 0x11, 0xFA, 0xA2, 0xCA, 0x06, 0xF3, 0xBC,
    0x5F, 0x8A, 0xCA, 0x17, 0x63, 0x36, 0x87, 0xCF, 0x8D, 0x5C, 0xA0, 0x56,
    0x84, 0x44, 0x61, 0xB2, 0x33, 0x42, 0x07, 0x58, 0x9F, 0x0C, 0x9E, 0x49,
    0x83,
};

// issuer:test cert
// subject:test cert
// issuerKey:secp256r1
// subjectKey:secp256r1
// serialNumber:2
std::vector<uint8_t> kTestCert2DER = {
    0x30, 0x82, 0x01, 0x1E, 0x30, 0x81, 0xC2, 0xA0, 0x03, 0x02, 0x01, 0x02,
    0x02, 0x01, 0x02, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7,
    0x0D, 0x01, 0x01, 0x0B, 0x05, 0x00, 0x30, 0x14, 0x31, 0x12, 0x30, 0x10,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x09, 0x74, 0x65, 0x73, 0x74, 0x20,
    0x63, 0x65, 0x72, 0x74, 0x30, 0x22, 0x18, 0x0F, 0x32, 0x30, 0x31, 0x37,
    0x31, 0x31, 0x32, 0x37, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5A, 0x18,
    0x0F, 0x32, 0x30, 0x32, 0x30, 0x30, 0x32, 0x30, 0x35, 0x30, 0x30, 0x30,
    0x30, 0x30, 0x30, 0x5A, 0x30, 0x14, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03,
    0x55, 0x04, 0x03, 0x0C, 0x09, 0x74, 0x65, 0x73, 0x74, 0x20, 0x63, 0x65,
    0x72, 0x74, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE,
    0x3D, 0x02, 0x01, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x03, 0x01,
    0x07, 0x03, 0x42, 0x00, 0x04, 0x4F, 0xBF, 0xBB, 0xBB, 0x61, 0xE0, 0xF8,
    0xF9, 0xB1, 0xA6, 0x0A, 0x59, 0xAC, 0x87, 0x04, 0xE2, 0xEC, 0x05, 0x0B,
    0x42, 0x3E, 0x3C, 0xF7, 0x2E, 0x92, 0x3F, 0x2C, 0x4F, 0x79, 0x4B, 0x45,
    0x5C, 0x2A, 0x69, 0xD2, 0x33, 0x45, 0x6C, 0x36, 0xC4, 0x11, 0x9D, 0x07,
    0x06, 0xE0, 0x0E, 0xED, 0xC8, 0xD1, 0x93, 0x90, 0xD7, 0x99, 0x1B, 0x7B,
    0x2D, 0x07, 0xA3, 0x04, 0xEA, 0xA0, 0x4A, 0xA6, 0xC0, 0x30, 0x0D, 0x06,
    0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x0B, 0x05, 0x00,
    0x03, 0x48, 0x00, 0x30, 0x45, 0x02, 0x20, 0x5C, 0x75, 0x51, 0x9F, 0x13,
    0x11, 0x50, 0xCD, 0x5D, 0x8A, 0xDE, 0x20, 0xA3, 0xBC, 0x06, 0x30, 0x91,
    0xFF, 0xB2, 0x73, 0x75, 0x5F, 0x31, 0x64, 0xEC, 0xFD, 0xCB, 0x42, 0x80,
    0x0A, 0x70, 0xE6, 0x02, 0x21, 0x00, 0xF6, 0x5E, 0x42, 0xC7, 0x54, 0x40,
    0x81, 0xE9, 0x4C, 0x16, 0x48, 0xB1, 0x39, 0x0A, 0xA0, 0xE2, 0x8C, 0x23,
    0xAA, 0xC5, 0xBB, 0xAC, 0xEB, 0x9B, 0x15, 0x0B, 0x2F, 0xB7, 0xF5, 0x85,
    0xB2, 0x54,
};

std::vector<uint8_t> kTestCertSubjectDER = {
    0x30, 0x14, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04, 0x03,
    0x0C, 0x09, 0x74, 0x65, 0x73, 0x74, 0x20, 0x63, 0x65, 0x72, 0x74,
};

// issuer:test cert
// subject:unrelated subject DN
// issuerKey:secp256r1
// subjectKey:secp256r1
// serialNumber:3
std::vector<uint8_t> kUnrelatedTestCertDER = {
    0x30, 0x82, 0x01, 0x28, 0x30, 0x81, 0xCD, 0xA0, 0x03, 0x02, 0x01, 0x02,
    0x02, 0x01, 0x03, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7,
    0x0D, 0x01, 0x01, 0x0B, 0x05, 0x00, 0x30, 0x14, 0x31, 0x12, 0x30, 0x10,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x09, 0x74, 0x65, 0x73, 0x74, 0x20,
    0x63, 0x65, 0x72, 0x74, 0x30, 0x22, 0x18, 0x0F, 0x32, 0x30, 0x31, 0x37,
    0x31, 0x31, 0x32, 0x37, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5A, 0x18,
    0x0F, 0x32, 0x30, 0x32, 0x30, 0x30, 0x32, 0x30, 0x35, 0x30, 0x30, 0x30,
    0x30, 0x30, 0x30, 0x5A, 0x30, 0x1F, 0x31, 0x1D, 0x30, 0x1B, 0x06, 0x03,
    0x55, 0x04, 0x03, 0x0C, 0x14, 0x75, 0x6E, 0x72, 0x65, 0x6C, 0x61, 0x74,
    0x65, 0x64, 0x20, 0x73, 0x75, 0x62, 0x6A, 0x65, 0x63, 0x74, 0x20, 0x44,
    0x4E, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D,
    0x02, 0x01, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x03, 0x01, 0x07,
    0x03, 0x42, 0x00, 0x04, 0x4F, 0xBF, 0xBB, 0xBB, 0x61, 0xE0, 0xF8, 0xF9,
    0xB1, 0xA6, 0x0A, 0x59, 0xAC, 0x87, 0x04, 0xE2, 0xEC, 0x05, 0x0B, 0x42,
    0x3E, 0x3C, 0xF7, 0x2E, 0x92, 0x3F, 0x2C, 0x4F, 0x79, 0x4B, 0x45, 0x5C,
    0x2A, 0x69, 0xD2, 0x33, 0x45, 0x6C, 0x36, 0xC4, 0x11, 0x9D, 0x07, 0x06,
    0xE0, 0x0E, 0xED, 0xC8, 0xD1, 0x93, 0x90, 0xD7, 0x99, 0x1B, 0x7B, 0x2D,
    0x07, 0xA3, 0x04, 0xEA, 0xA0, 0x4A, 0xA6, 0xC0, 0x30, 0x0D, 0x06, 0x09,
    0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x0B, 0x05, 0x00, 0x03,
    0x47, 0x00, 0x30, 0x44, 0x02, 0x20, 0x5C, 0x75, 0x51, 0x9F, 0x13, 0x11,
    0x50, 0xCD, 0x5D, 0x8A, 0xDE, 0x20, 0xA3, 0xBC, 0x06, 0x30, 0x91, 0xFF,
    0xB2, 0x73, 0x75, 0x5F, 0x31, 0x64, 0xEC, 0xFD, 0xCB, 0x42, 0x80, 0x0A,
    0x70, 0xE6, 0x02, 0x20, 0x0F, 0x1A, 0x04, 0xC2, 0xF8, 0xBA, 0xC2, 0x94,
    0x26, 0x6E, 0xBC, 0x91, 0x7D, 0xDB, 0x75, 0x7B, 0xE8, 0xA3, 0x4F, 0x69,
    0x1B, 0xF3, 0x1F, 0x2C, 0xCE, 0x82, 0x67, 0xC9, 0x5B, 0xBB, 0xBA, 0x0A,
};

class PK11FindRawCertsBySubjectTest : public ::testing::Test {
 protected:
  PK11FindRawCertsBySubjectTest()
      : mSlot(nullptr), mTestCertDBDir("PK11FindRawCertsBySubjectTest-") {}

  virtual void SetUp() {
    std::string testCertDBPath(mTestCertDBDir.GetPath());
    const char* testName =
        ::testing::UnitTest::GetInstance()->current_test_info()->name();
    std::string modspec = "configDir='sql:";
    modspec.append(testCertDBPath);
    modspec.append("' tokenDescription='");
    modspec.append(testName);
    modspec.append("'");
    mSlot = SECMOD_OpenUserDB(modspec.c_str());
    ASSERT_NE(mSlot, nullptr);
  }

  virtual void TearDown() {
    ASSERT_EQ(SECMOD_CloseUserDB(mSlot), SECSuccess);
    PK11_FreeSlot(mSlot);
    std::string testCertDBPath(mTestCertDBDir.GetPath());
    ASSERT_EQ(0, unlink((testCertDBPath + "/cert9.db").c_str()));
    ASSERT_EQ(0, unlink((testCertDBPath + "/key4.db").c_str()));
  }

  PK11SlotInfo* mSlot;
  ScopedUniqueDirectory mTestCertDBDir;
};

// If we don't have any certificates, we shouldn't get any when we search for
// them.
TEST_F(PK11FindRawCertsBySubjectTest, TestNoCertsImportedNoCertsFound) {
  SECItem subjectItem = {siBuffer,
                         const_cast<unsigned char*>(kTestCertSubjectDER.data()),
                         (unsigned int)kTestCertSubjectDER.size()};
  CERTCertificateList* certificates = nullptr;
  SECStatus rv =
      PK11_FindRawCertsWithSubject(mSlot, &subjectItem, &certificates);
  EXPECT_EQ(rv, SECSuccess);
  EXPECT_EQ(certificates, nullptr);
}

// If we have one certificate but it has an unrelated subject DN, we shouldn't
// get it when we search.
TEST_F(PK11FindRawCertsBySubjectTest, TestOneCertImportedNoCertsFound) {
  char certNickname[] = "Unrelated Cert";
  SECItem certItem = {siBuffer,
                      const_cast<unsigned char*>(kUnrelatedTestCertDER.data()),
                      (unsigned int)kUnrelatedTestCertDER.size()};
  ASSERT_EQ(PK11_ImportDERCert(mSlot, &certItem, CK_INVALID_HANDLE,
                               certNickname, false),
            SECSuccess);

  SECItem subjectItem = {siBuffer,
                         const_cast<unsigned char*>(kTestCertSubjectDER.data()),
                         (unsigned int)kTestCertSubjectDER.size()};
  CERTCertificateList* certificates = nullptr;
  SECStatus rv =
      PK11_FindRawCertsWithSubject(mSlot, &subjectItem, &certificates);
  EXPECT_EQ(rv, SECSuccess);
  EXPECT_EQ(certificates, nullptr);
}

TEST_F(PK11FindRawCertsBySubjectTest, TestMultipleMatchingCertsFound) {
  char cert1Nickname[] = "Test Cert 1";
  SECItem cert1Item = {siBuffer,
                       const_cast<unsigned char*>(kTestCert1DER.data()),
                       (unsigned int)kTestCert1DER.size()};
  ASSERT_EQ(PK11_ImportDERCert(mSlot, &cert1Item, CK_INVALID_HANDLE,
                               cert1Nickname, false),
            SECSuccess);
  char cert2Nickname[] = "Test Cert 2";
  SECItem cert2Item = {siBuffer,
                       const_cast<unsigned char*>(kTestCert2DER.data()),
                       (unsigned int)kTestCert2DER.size()};
  ASSERT_EQ(PK11_ImportDERCert(mSlot, &cert2Item, CK_INVALID_HANDLE,
                               cert2Nickname, false),
            SECSuccess);
  char unrelatedCertNickname[] = "Unrelated Test Cert";
  SECItem unrelatedCertItem = {
      siBuffer, const_cast<unsigned char*>(kUnrelatedTestCertDER.data()),
      (unsigned int)kUnrelatedTestCertDER.size()};
  ASSERT_EQ(PK11_ImportDERCert(mSlot, &unrelatedCertItem, CK_INVALID_HANDLE,
                               unrelatedCertNickname, false),
            SECSuccess);

  CERTCertificateList* certificates = nullptr;
  SECItem subjectItem = {siBuffer,
                         const_cast<unsigned char*>(kTestCertSubjectDER.data()),
                         (unsigned int)kTestCertSubjectDER.size()};
  SECStatus rv =
      PK11_FindRawCertsWithSubject(mSlot, &subjectItem, &certificates);
  EXPECT_EQ(rv, SECSuccess);
  ASSERT_NE(certificates, nullptr);
  ScopedCERTCertificateList scopedCertificates(certificates);
  ASSERT_EQ(scopedCertificates->len, 2);

  std::vector<uint8_t> foundCert1(
      scopedCertificates->certs[0].data,
      scopedCertificates->certs[0].data + scopedCertificates->certs[0].len);
  std::vector<uint8_t> foundCert2(
      scopedCertificates->certs[1].data,
      scopedCertificates->certs[1].data + scopedCertificates->certs[1].len);
  EXPECT_TRUE(foundCert1 == kTestCert1DER || foundCert1 == kTestCert2DER);
  EXPECT_TRUE(foundCert2 == kTestCert1DER || foundCert2 == kTestCert2DER);
  EXPECT_TRUE(foundCert1 != foundCert2);
}

// If we try to search the internal slots, we won't find the certificate we just
// imported (because it's on a different slot).
TEST_F(PK11FindRawCertsBySubjectTest, TestNoCertsOnInternalSlots) {
  char cert1Nickname[] = "Test Cert 1";
  SECItem cert1Item = {siBuffer,
                       const_cast<unsigned char*>(kTestCert1DER.data()),
                       (unsigned int)kTestCert1DER.size()};
  ASSERT_EQ(PK11_ImportDERCert(mSlot, &cert1Item, CK_INVALID_HANDLE,
                               cert1Nickname, false),
            SECSuccess);

  SECItem subjectItem = {siBuffer,
                         const_cast<unsigned char*>(kTestCertSubjectDER.data()),
                         (unsigned int)kTestCertSubjectDER.size()};
  CERTCertificateList* internalKeySlotCertificates = nullptr;
  ScopedPK11SlotInfo internalKeySlot(PK11_GetInternalKeySlot());
  SECStatus rv = PK11_FindRawCertsWithSubject(
      internalKeySlot.get(), &subjectItem, &internalKeySlotCertificates);
  EXPECT_EQ(rv, SECSuccess);
  EXPECT_EQ(internalKeySlotCertificates, nullptr);

  CERTCertificateList* internalSlotCertificates = nullptr;
  ScopedPK11SlotInfo internalSlot(PK11_GetInternalSlot());
  rv = PK11_FindRawCertsWithSubject(internalSlot.get(), &subjectItem,
                                    &internalSlotCertificates);
  EXPECT_EQ(rv, SECSuccess);
  EXPECT_EQ(internalSlotCertificates, nullptr);
}

// issuer:test cert
// subject:(empty - this had to be done by hand as pycert doesn't support this)
// issuerKey:secp256r1
// subjectKey:secp256r1
// serialNumber:4
std::vector<uint8_t> kEmptySubjectCertDER = {
    0x30, 0x82, 0x01, 0x09, 0x30, 0x81, 0xAE, 0xA0, 0x03, 0x02, 0x01, 0x02,
    0x02, 0x01, 0x04, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7,
    0x0D, 0x01, 0x01, 0x0B, 0x05, 0x00, 0x30, 0x14, 0x31, 0x12, 0x30, 0x10,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x09, 0x74, 0x65, 0x73, 0x74, 0x20,
    0x63, 0x65, 0x72, 0x74, 0x30, 0x22, 0x18, 0x0F, 0x32, 0x30, 0x31, 0x37,
    0x31, 0x31, 0x32, 0x37, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5A, 0x18,
    0x0F, 0x32, 0x30, 0x32, 0x30, 0x30, 0x32, 0x30, 0x35, 0x30, 0x30, 0x30,
    0x30, 0x30, 0x30, 0x5A, 0x30, 0x00, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07,
    0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x08, 0x2A, 0x86, 0x48,
    0xCE, 0x3D, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x4F, 0xBF, 0xBB,
    0xBB, 0x61, 0xE0, 0xF8, 0xF9, 0xB1, 0xA6, 0x0A, 0x59, 0xAC, 0x87, 0x04,
    0xE2, 0xEC, 0x05, 0x0B, 0x42, 0x3E, 0x3C, 0xF7, 0x2E, 0x92, 0x3F, 0x2C,
    0x4F, 0x79, 0x4B, 0x45, 0x5C, 0x2A, 0x69, 0xD2, 0x33, 0x45, 0x6C, 0x36,
    0xC4, 0x11, 0x9D, 0x07, 0x06, 0xE0, 0x0E, 0xED, 0xC8, 0xD1, 0x93, 0x90,
    0xD7, 0x99, 0x1B, 0x7B, 0x2D, 0x07, 0xA3, 0x04, 0xEA, 0xA0, 0x4A, 0xA6,
    0xC0, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01,
    0x01, 0x0B, 0x05, 0x00, 0x03, 0x47, 0x00, 0x30, 0x44, 0x02, 0x20, 0x5C,
    0x75, 0x51, 0x9F, 0x13, 0x11, 0x50, 0xCD, 0x5D, 0x8A, 0xDE, 0x20, 0xA3,
    0xBC, 0x06, 0x30, 0x91, 0xFF, 0xB2, 0x73, 0x75, 0x5F, 0x31, 0x64, 0xEC,
    0xFD, 0xCB, 0x42, 0x80, 0x0A, 0x70, 0xE6, 0x02, 0x20, 0x31, 0x1B, 0x92,
    0xAA, 0xA8, 0xB7, 0x51, 0x52, 0x7B, 0x64, 0xD6, 0xF7, 0x2F, 0x0C, 0xFB,
    0xBB, 0xD5, 0xDF, 0x86, 0xA3, 0x97, 0x96, 0x60, 0x42, 0xDA, 0xD4, 0xA8,
    0x5F, 0x2F, 0xA4, 0xDE, 0x7C};

std::vector<uint8_t> kEmptySubjectDER = {0x30, 0x00};

// This certificate has the smallest possible subject. Finding it should work.
TEST_F(PK11FindRawCertsBySubjectTest, TestFindEmptySubject) {
  char emptySubjectCertNickname[] = "Empty Subject Cert";
  SECItem emptySubjectCertItem = {
      siBuffer, const_cast<unsigned char*>(kEmptySubjectCertDER.data()),
      (unsigned int)kEmptySubjectCertDER.size()};
  ASSERT_EQ(PK11_ImportDERCert(mSlot, &emptySubjectCertItem, CK_INVALID_HANDLE,
                               emptySubjectCertNickname, false),
            SECSuccess);

  SECItem subjectItem = {siBuffer,
                         const_cast<unsigned char*>(kEmptySubjectDER.data()),
                         (unsigned int)kEmptySubjectDER.size()};
  CERTCertificateList* certificates = nullptr;
  SECStatus rv =
      PK11_FindRawCertsWithSubject(mSlot, &subjectItem, &certificates);
  EXPECT_EQ(rv, SECSuccess);
  ASSERT_NE(certificates, nullptr);
  ScopedCERTCertificateList scopedCertificates(certificates);
  ASSERT_EQ(scopedCertificates->len, 1);

  std::vector<uint8_t> foundCert(
      scopedCertificates->certs[0].data,
      scopedCertificates->certs[0].data + scopedCertificates->certs[0].len);
  EXPECT_EQ(foundCert, kEmptySubjectCertDER);
}

// Searching for a zero-length subject doesn't make sense (the minimum subject
// is the SEQUENCE tag followed by a length byte of 0), but it shouldn't cause
// problems.
TEST_F(PK11FindRawCertsBySubjectTest, TestSearchForNullSubject) {
  char cert1Nickname[] = "Test Cert 1";
  SECItem cert1Item = {siBuffer,
                       const_cast<unsigned char*>(kTestCert1DER.data()),
                       (unsigned int)kTestCert1DER.size()};
  ASSERT_EQ(PK11_ImportDERCert(mSlot, &cert1Item, CK_INVALID_HANDLE,
                               cert1Nickname, false),
            SECSuccess);

  SECItem subjectItem = {siBuffer, nullptr, 0};
  CERTCertificateList* certificates = nullptr;
  SECStatus rv =
      PK11_FindRawCertsWithSubject(mSlot, &subjectItem, &certificates);
  EXPECT_EQ(rv, SECSuccess);
  EXPECT_EQ(certificates, nullptr);
}

}  // namespace nss_test
