package org.htmlcleaner;

import java.io.IOException;

import org.junit.Test;

import junit.framework.TestCase;

/**
 * Tests that tag closed due to one of its children (when the child tag is not allowed to be inside parent) is then
 * reopened.
 * Examples:
 * <pre>
 * <div><p>text1<table><tr><td>text2</td></tr></table>text3</p></div>
 * </pre>
 * table is not allowed inside a <p> most browsers handle this by placing the table close to line before and line after and in general allowing it.
 * 
 * Cleaning here normally would result in :
 * <pre>
 * <div><p>text1<table><tr><td>text2</td></tr></table>text3</div>
 * </pre>
 * 'text3' is no longer inside the original element type ( 'p' ). Instead 'text3' is now within a 'div'. 
 * text3 would no longer be styled correctly.
 * 
 * A more correct result is:
 * <pre>
 * <div><p>text1<table><tr><td>text2</td></tr></table><p>text3</p></div>
 * </pre>
 */ 
public class ClosedTagReopenTest extends TestCase {

    public void testSimpleHTML4() throws IOException {
        CleanerProperties properties = new CleanerProperties();
        properties.setHtmlVersion(HtmlCleaner.HTML_4);
        properties.setOmitXmlDeclaration(true);
        properties.setOmitHtmlEnvelope(true);
        SimpleXmlSerializer serializer = new SimpleXmlSerializer(properties);
        String[][] tests= {
            new String[] { "<p>text1<table><tr><td>text2</td></tr></table>text3</p>", "<p>text1</p><table><tbody><tr><td>text2</td></tr></tbody></table><p>text3</p>" },
            new String[] {"</p>text1","text1"},
            new String[] {"<p>text1<div>text2</div>text3</p>", "<p>text1</p><div>text2</div><p>text3</p>"},
            new String[] { "<div>text1<p>text2</p>text3</div>", "<div>text1<p>text2</p>text3</div>"},
            new String[] {"<font>text1<p>text2</p>text3</font>", "<font>text1</font><p><font>text2</font></p><font>text3</font>"},
            new String[] {"<p>text1<div>text2</div>text3<div>text4</div></p>", "<p>text1</p><div>text2</div><p>text3</p><div>text4</div>"},
            new String[] {"<p>text1<div>text2</div></p>", "<p>text1</p><div>text2</div>"},     
            new String[] {"<p>text1<p>text2</p></p>", "<p>text1</p><p>text2</p>"},
            //test multiple internal breaks
            new String[] {"<p><div>text1<p>text2<div>text3<p>text4<div>text5</div></p></div></p></div>","<p></p><div>text1<p>text2</p><div>text3<p>text4</p><div>text5</div></div></div>"},
            // test attribute preservation
            new String[] { "<p class=\"p_class\" random=\"attribute\">text1<table><tr><td>text2</td></tr></table>text3</p>", 
            "<p class=\"p_class\" random=\"attribute\">text1</p><table><tbody><tr><td>text2</td></tr></tbody></table><p class=\"p_class\" random=\"attribute\">text3</p>" },
            // but not all attributes ( id attribute must be unique ) 
            // TODO: maybe a generated id so that correlation can be found?
            new String[] { "<p class=\"p_class\" random=\"attribute\" id=\"just_me\">text1<table><tr><td>text2</td></tr></table>text3</p>", 
            "<p class=\"p_class\" random=\"attribute\" id=\"just_me\">text1</p><table><tbody><tr><td>text2</td></tr></tbody></table><p class=\"p_class\" random=\"attribute\">text3</p>" },
            // test multiple replacements
            // test to see if nested good <p> can be handled.
            new String[] { "<p class=\"p_class\" random=\"attribute\">text1<table><tr><td>text2<p>text2a</p></td></tr></table>text3<ul><li>text4</ul>text5<ul><li>text6</ul></p>", 
                "<p class=\"p_class\" random=\"attribute\">text1</p><table><tbody><tr><td>text2<p>text2a</p></td></tr></tbody></table>" +
                "<p class=\"p_class\" random=\"attribute\">text3</p>" +
                "<ul><li>text4</li></ul>" +
                "<p class=\"p_class\" random=\"attribute\">text5</p>" +
            "<ul><li>text6</li></ul>" },
            new String[] { "<p class=\"p_class\" random=\"attribute\">text1<table><tr><td>text2<p class=\"another_p_element\">text2a<div>test2b</div>test2c</p></td></tr></table>text3<ul><li>text4</ul>text5<ul><li>text6</ul></p>", 
                "<p class=\"p_class\" random=\"attribute\">text1</p><table><tbody><tr><td>text2<p class=\"another_p_element\">text2a</p><div>test2b</div><p class=\"another_p_element\">test2c</p></td></tr></tbody></table>" +
                "<p class=\"p_class\" random=\"attribute\">text3</p>" +
                "<ul><li>text4</li></ul>" +
                "<p class=\"p_class\" random=\"attribute\">text5</p>" +
            "<ul><li>text6</li></ul>" },
            new String[]{"<p>text1<table><tr><td>text2<tr><td>text3</table>text4</p>","<p>text1</p><table><tbody><tr><td>text2</td></tr><tr><td>text3</td></tr></tbody></table><p>text4</p>"}
        };
        for(String[] test: tests) {
            String cleaned = serializer.getAsString(test[0]);
            assertEquals("started with="+test[0], test[1], cleaned);
        }
    }
    
    @Test
    public void testSimpleHTML5() throws IOException {
    	 CleanerProperties properties = new CleanerProperties();
         properties.setHtmlVersion(HtmlCleaner.HTML_5);
         properties.setOmitXmlDeclaration(true);
         properties.setOmitHtmlEnvelope(true);
         SimpleXmlSerializer serializer = new SimpleXmlSerializer(properties);
        String[][] tests= {
            new String[] { "<p>text1<table><tr><td>text2</td></tr></table>text3</p>", "<p>text1</p><table><tbody><tr><td>text2</td></tr></tbody></table><p>text3</p>" },
            new String[] {"</p>text1","text1"},
            new String[] {"<p>text1<div>text2</div>text3</p>", "<p>text1</p><div>text2</div><p>text3</p>"},
            new String[] { "<div>text1<p>text2</p>text3</div>", "<div>text1<p>text2</p>text3</div>"},
            new String[] {"text1<p>text2</p>text3", "text1<p>text2</p>text3"},
            new String[] {"<p>text1<div>text2</div>text3<div>text4</div></p>", "<p>text1</p><div>text2</div><p>text3</p><div>text4</div>"},
            new String[] {"<p>text1<div>text2</div></p>", "<p>text1</p><div>text2</div>"},     
            new String[] {"<p>text1<p>text2</p></p>", "<p>text1</p><p>text2</p>"},
            //test multiple internal breaks
            new String[] {"<p><div>text1<p>text2<div>text3<p>text4<div>text5</div></p></div></p></div>","<p></p><div>text1<p>text2</p><div>text3<p>text4</p><div>text5</div></div></div>"},
            // test attribute preservation
            new String[] { "<p class=\"p_class\" random=\"attribute\">text1<table><tr><td>text2</td></tr></table>text3</p>", 
            "<p class=\"p_class\" random=\"attribute\">text1</p><table><tbody><tr><td>text2</td></tr></tbody></table><p class=\"p_class\" random=\"attribute\">text3</p>" },
            // but not all attributes ( id attribute must be unique ) 
            // TODO: maybe a generated id so that correlation can be found?
            new String[] { "<p class=\"p_class\" random=\"attribute\" id=\"just_me\">text1<table><tr><td>text2</td></tr></table>text3</p>", 
            "<p class=\"p_class\" random=\"attribute\" id=\"just_me\">text1</p><table><tbody><tr><td>text2</td></tr></tbody></table><p class=\"p_class\" random=\"attribute\">text3</p>" },
            // test multiple replacements
            // test to see if nested good <p> can be handled.
            new String[] { "<p class=\"p_class\" random=\"attribute\">text1<table><tr><td>text2<p>text2a</p></td></tr></table>text3<ul><li>text4</ul>text5<ul><li>text6</ul></p>", 
                "<p class=\"p_class\" random=\"attribute\">text1</p><table><tbody><tr><td>text2<p>text2a</p></td></tr></tbody></table>" +
                "<p class=\"p_class\" random=\"attribute\">text3</p>" +
                "<ul><li>text4</li></ul>" +
                "<p class=\"p_class\" random=\"attribute\">text5</p>" +
            "<ul><li>text6</li></ul>" },
            new String[] { "<p class=\"p_class\" random=\"attribute\">text1<table><tr><td>text2<p class=\"another_p_element\">text2a<div>test2b</div>test2c</p></td></tr></table>text3<ul><li>text4</ul>text5<ul><li>text6</ul></p>", 
                "<p class=\"p_class\" random=\"attribute\">text1</p><table><tbody><tr><td>text2<p class=\"another_p_element\">text2a</p><div>test2b</div><p class=\"another_p_element\">test2c</p></td></tr></tbody></table>" +
                "<p class=\"p_class\" random=\"attribute\">text3</p>" +
                "<ul><li>text4</li></ul>" +
                "<p class=\"p_class\" random=\"attribute\">text5</p>" +
            "<ul><li>text6</li></ul>" },
            new String[]{"<p>text1<table><tr><td>text2<tr><td>text3</table>text4</p>","<p>text1</p><table><tbody><tr><td>text2</td></tr><tr><td>text3</td></tr></tbody></table><p>text4</p>"}
        };
        for(String[] test: tests) {
            String cleaned = serializer.getAsString(test[0]);
            assertEquals("started with="+test[0], test[1], cleaned);
        }
    }
    
}
