/**
 * Copyright (C) 1999, 2000, 2001, 2002  Free Software Foundation, Inc.
 *
 * This file is part of GNU gengetopt 
 *
 * GNU gengetopt is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option) 
 * any later version.
 *
 * GNU gengetopt is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details. 
 *
 * You should have received a copy of the GNU General Public License along 
 * with gengetopt; see the file COPYING. If not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <string>

#include <fstream>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

extern "C"
{
#include "argsdef.h"
#include "global_opts.h"
};

#include "ggo_options.h"

#include "gm.h"
#include "my_sstream.h"

#include "groups.h"
#include "string_opt_arg.h"
#include "string_opt_arg_free.h"
#include "int_opt_arg.h"
#include "long_opt_arg.h"
#include "long_long_opt_arg.h"
#include "float_opt_arg.h"
#include "flag_opt_arg.h"
#include "required_option.h"
#include "generic_option.h"
#include "group_option.h"
#include "group_counter.h"
#include "generic_option_group.h"
#include "handle_help.h"
#include "handle_version.h"
#include "no_short_option.h"
#include "handle_unamed.h"
#include "print_group.h"
#include "config_parser_header.h"
#include "multiple_opt_list.h"
#include "multiple_option.h"
#include "multiple_option_no_short.h"
#include "multiple_fill_array.h"
#include "multiple_fill_array_default.h"
#include "gen_strdup.h"

#include "conf_parser_gen.h"

#ifndef HAVE_STRDUP
extern "C" char *strdup (const char *s) ;
#endif

using std::ofstream;
using std::endl;
using std::string;

extern char * gengetopt_package;
extern char * gengetopt_version;
extern char * gengetopt_purpose;
extern char * gengetopt_input_filename;

extern groups_collection_t gengetopt_groups;

static char *create_filename (char *name, char *ext);
static ofstream *open_fstream (char *filename);
static char *canonize_names(const char * name);
static const string canonize_name(const string &name);
static const string strip_path(const string &);
static const string to_upper(const string &);
static void _generate_option_arg(ostream &stream, 
                                 unsigned int indent,
                                 struct gengetopt_option * opt);
static void 
generate_help_desc_print(ostream &stream, 
			 unsigned int desc_column,
			 const char *descript, const char *defval);

char *
create_filename (char *name, char *ext)
{
  char *filename ;

  filename = (char *) malloc (strlen (name) + strlen (ext) + 2);
  /* 2 = 1 for the . and one for the '\0' */
  if (! filename)
    {
      fprintf (stderr, "Error in memory allocation! %s %d\n",
               __FILE__, __LINE__);
      abort ();
    }

  sprintf (filename, "%s.%s", name, ext);

  return filename ;
}

ofstream *
open_fstream (char *filename)
{
  ofstream *fstream = new ofstream (filename);
  
  if ( ! (*fstream) )
    {
      fprintf( stderr, "Error creating %s\n", filename ) ;
      abort() ;
    }

  return fstream;
}

CmdlineParserCreator::CmdlineParserCreator (char *function_name, 
                                            bool unamed_options_,
                                            char *filename_, 
                                            char *header_ext, char *c_ext,
                                            bool long_help_, 
                                            bool no_handle_help_,
                                            bool no_handle_version_,
                                            bool no_handle_error_,
                                            bool conf_parser_,
                                            const string &comment_) :
  filename (filename_), 
  comment (comment_),
  unamed_options (unamed_options_), 
  long_help (long_help_), no_handle_help (no_handle_help_),
  no_handle_version (no_handle_version_), no_handle_error (no_handle_error_),
  conf_parser (conf_parser_),
  tab_indentation (0)
{
  parser_function_name = canonize_names (function_name);
  c_filename = create_filename (filename, c_ext);
  header_filename = create_filename (filename, header_ext);

  // header_gen_class
  const string stripped_header_file_name = strip_path (filename);
  set_header_file_name (stripped_header_file_name);
  header_gen_class::set_generator_version (VERSION);
  const string my_ifndefname = 
    to_upper (strip_path (stripped_header_file_name));
  set_ifndefname (canonize_names (my_ifndefname.c_str ()));
  header_gen_class::set_parser_name (parser_function_name);
  const string my_package_var_name = 
    to_upper (parser_function_name) + "_PACKAGE";
  const string my_version_var_name = 
    to_upper (parser_function_name) + "_VERSION";
  header_gen_class::set_package_var_name (my_package_var_name);
  c_source_gen_class::set_package_var_name (my_package_var_name);
  header_gen_class::set_version_var_name (my_version_var_name);
  c_source_gen_class::set_version_var_name (my_version_var_name);
  const string uppersand = "\"";

  if (gengetopt_package)
    set_package_var_val
      (uppersand + gengetopt_package + uppersand);
  else
    set_package_var_val ("PACKAGE");

  if (gengetopt_version)
    set_version_var_val
      (uppersand + gengetopt_version + uppersand);
  else
    set_version_var_val ("VERSION");

  if (conf_parser)
    {
      // the config file parser prototype
      ostringstream conf_parser_proto;
      config_parser_header_gen_class cf_gen;
      cf_gen.set_parser_name (parser_function_name);
      conf_parser_proto << endl;
      cf_gen.generate_config_parser_header (conf_parser_proto);
      conf_parser_proto << endl;
      set_config_parser_proto (conf_parser_proto.str ());
    }

  // c_source_gen_class
  set_command_line (comment);
  c_source_gen_class::set_generator_version (VERSION);
  c_source_gen_class::set_parser_name (parser_function_name);
  set_source_name (filename);

  set_gen_exit (no_handle_error ? "return" : "exit");
}

void
CmdlineParserCreator::generateBreak(ostream &stream, unsigned int indent)
{
  string indent_str (indent, ' ');
  
  stream << endl;
  stream << indent_str;
  stream << "break;";
}

void
CmdlineParserCreator::do_update_arg (struct gengetopt_option *opt,
                                     ostream &stream,
                                     unsigned int indent) 
{
  const string argstr ("optarg");
  string_opt_arg_gen_class str_gen;
  str_gen.set_optarg (argstr);
  string_opt_arg_free_gen_class str_gen_free;
  int_opt_arg_gen_class int_gen;
  int_gen.set_optarg (argstr);
  long_opt_arg_gen_class long_gen;
  long_gen.set_optarg (argstr);
  long_long_opt_arg_gen_class long_long_gen;
  long_long_gen.set_optarg (argstr);
  float_opt_arg_gen_class float_gen;
  float_gen.set_optarg (argstr);
  flag_opt_arg_gen_class flag_gen;

  if (opt->multiple)
    {
      int_gen.set_structure (string (opt->var_arg) + "_new");
      str_gen.set_structure (string (opt->var_arg) + "_new");
      long_gen.set_structure (string (opt->var_arg) + "_new");
      long_long_gen.set_structure (string (opt->var_arg) + "_new");
      float_gen.set_structure (string (opt->var_arg) + "_new");
    }
  else
    {
      int_gen.set_structure (ARGS_STRUCT);
      str_gen.set_structure (ARGS_STRUCT);
      str_gen_free.set_structure (ARGS_STRUCT);
      long_gen.set_structure (ARGS_STRUCT);
      long_long_gen.set_structure (ARGS_STRUCT);
      float_gen.set_structure (ARGS_STRUCT);
    }

  switch (opt->type)
    {
    case ARG_NO:
      stream << "break;";
      break;
    case ARG_FLAG:
      flag_gen.set_opt_var (opt->var_arg);
      flag_gen.generate_flag_opt_arg (stream, indent);
      generateBreak(stream, indent);
      break;
    case ARG_STRING:
      str_gen.set_opt_var (opt->var_arg);
      if (! opt->multiple && opt->default_given)
        {
          str_gen_free.set_opt_var (opt->var_arg);
          str_gen_free.generate_string_opt_arg_free (stream, indent);
        }
      str_gen.generate_string_opt_arg (stream, indent);
      generateBreak(stream, indent);
      break;
    case ARG_INT:
      int_gen.set_cast ("");
      int_gen.set_opt_var (opt->var_arg);
      int_gen.generate_int_opt_arg (stream, indent);
      generateBreak(stream, indent);
      break;
    case ARG_SHORT:
      int_gen.set_cast ("(short)");
      int_gen.set_opt_var (opt->var_arg);
      int_gen.generate_int_opt_arg (stream, indent);
      generateBreak(stream, indent);
      break;
    case ARG_LONG:
      long_gen.set_cast ("");
      long_gen.set_opt_var (opt->var_arg);
      long_gen.generate_long_opt_arg (stream, indent);
      generateBreak(stream, indent);
      break;
    case ARG_FLOAT:
      float_gen.set_cast ("(float)");
      float_gen.set_opt_var (opt->var_arg);
      float_gen.generate_float_opt_arg (stream, indent);
      generateBreak(stream, indent);
      break;
    case ARG_DOUBLE:
      float_gen.set_cast ("");
      float_gen.set_opt_var (opt->var_arg);
      float_gen.generate_float_opt_arg (stream, indent);
      generateBreak(stream, indent);
      break;
    case ARG_LONGDOUBLE:
      float_gen.set_cast ("(long double)");
      float_gen.set_opt_var (opt->var_arg);
      float_gen.generate_float_opt_arg (stream, indent);
      generateBreak(stream, indent);
      break;
    case ARG_LONGLONG:
      long_long_gen.set_opt_var (opt->var_arg);
      long_long_gen.generate_long_long_opt_arg (stream, indent);
      generateBreak(stream, indent);
      break;
    default:
      fprintf (stderr, "gengetopt: bug found in %s:%d\n", __FILE__,
               __LINE__);
      abort ();
    }
}

int
CmdlineParserCreator::generate ()
{
  int head_result;

  head_result = generate_header_file ();
  if (head_result)
    return head_result;

  return generate_source ();
}

int
CmdlineParserCreator::generate_header_file ()
{
  if (gengetopt_options == 0) 
    {
      fprintf (stderr, "gengetopt: none option given\n");
      return 1;
    }

  /* ****************************************************** */
  /* HEADER FILE******************************************* */
  /* ****************************************************** */

  ofstream *output_file = open_fstream (header_filename);
  generate_header (*output_file);
  output_file->close ();
  delete output_file;

  return 0;
}

void 
CmdlineParserCreator::generate_option_arg(ostream &stream, 
                                          unsigned int indent)
{
  struct gengetopt_option * opt;
  string indent_str (indent, ' ');
  bool first = true;

  foropt
    if (opt->type != ARG_NO) {
      switch (opt->type) {
      case ARG_FLAG:
      case ARG_STRING:
      case ARG_INT:
      case ARG_SHORT:
      case ARG_LONG:
      case ARG_FLOAT:
      case ARG_DOUBLE:
      case ARG_LONGDOUBLE:
      case ARG_LONGLONG:
        if (! first)
          stream << indent_str;
        else
          first = false;

        _generate_option_arg (stream, indent, opt);

        break;
      default: 
        fprintf (stderr, "gengetopt: bug found in %s:%d!!\n",
                 __FILE__, __LINE__);
        abort ();
      }
    }
}

void 
_generate_option_arg(ostream &stream, 
                     unsigned int indent,
                     struct gengetopt_option *opt)
{
  ostringstream opt_stream;

  if (opt->multiple)
    opt_stream << "* ";
  else
    opt_stream << " ";
      
  if (opt->type == ARG_FLAG)
    opt_stream << opt->var_arg << "_flag";
  else
    opt_stream << opt->var_arg << "_arg";
  
  opt_stream << ";\t/* " << opt->desc;
      
  if (opt->default_given)
    {
      opt_stream << " (default='" << opt->default_string << "')";
    }
      
  if (opt->type == ARG_FLAG)
    {
      if (opt->flagstat)
        opt_stream << " (default=on)";
      else
        opt_stream << " (default=off)";
    }
      
  opt_stream << ".  */\n";
  const string opt_string = opt_stream.str ();

  if (opt->type == ARG_LONGLONG)
    {
      string indent_str (indent, ' ');

      stream << "#ifdef HAVE_LONG_LONG\n";
      stream << indent_str;
      stream << arg_types[opt->type] << opt_string;
      stream << indent_str;
      stream << "#else\n";
      stream << indent_str;
      stream << arg_types[ARG_LONG] << opt_string;
      stream << indent_str;
      stream << "#endif\n";
    }
  else
    stream << arg_types[opt->type] << opt_string;
}

void 
CmdlineParserCreator::generate_option_given(ostream &stream, 
                                            unsigned int indent)
{
  struct gengetopt_option * opt;
  string indent_str (indent, ' ');
  bool first = true;

  foropt
    {
      switch (opt->type) {
      case ARG_NO:
      case ARG_FLAG:
      case ARG_STRING:
      case ARG_INT:
      case ARG_SHORT:
      case ARG_LONG:
      case ARG_FLOAT:
      case ARG_DOUBLE:
      case ARG_LONGDOUBLE:
      case ARG_LONGLONG: break;
      default:
        fprintf (stderr, "gengetopt: bug found in %s:%d!!\n",
                 __FILE__, __LINE__);
        abort ();
      }
      if (! first)
        stream << indent_str;
      else
        first = false;

      stream << "int " << opt->var_arg << 
        "_given ;\t/* Whether " << opt->long_opt << " was given.  */\n";
    }

  if (unamed_options)
    {
      stream << endl;
      stream << indent_str;
      stream << "char **inputs ; /* unamed options */\n" ;
      stream << indent_str;
      stream << "unsigned inputs_num ; /* unamed options number */" ;
    }
}

void 
CmdlineParserCreator::generate_print_purpose(ostream &stream, 
                                             unsigned int indent)
{
  if (gengetopt_purpose != NULL) 
    {
      string indent_str (indent, ' ');

      char *ptr;
      ptr = gengetopt_purpose;
      stream << indent_str;
      stream << "\"Purpose:\\n\"";
      stream << endl;
      stream << indent_str;
      stream << "\"  ";
      for (; *ptr!='\0'; ptr++) 
        {
          if (*ptr == '\n') 
            {
              stream << "\\n\"";
              stream << endl;
              stream << indent_str;
              stream << "\"  ";
            } else {
              stream << *ptr;
            }
        }
      stream << "\\n\""; // close the last line
      stream << endl;
      stream << indent_str;
      stream << "\"\\n\"\n"; // generate the last \n
    }
}

string
CmdlineParserCreator::generate_usage_string()
{
  struct gengetopt_option * opt;
  ostringstream usage;
  const char   *type_str;

  usage << "Usage: %s ";

  if ( long_help )
    {
      foropt
	if (opt->required) /* required options */
          switch (opt->type) {
          case ARG_INT:
          case ARG_SHORT:
          case ARG_LONG:
          case ARG_FLOAT:
          case ARG_DOUBLE:
          case ARG_LONGDOUBLE:
          case ARG_LONGLONG:
          case ARG_STRING: 
            if (opt->type_str)
              type_str = opt->type_str;
            else
              type_str = arg_names[opt->type];

            if (opt->short_opt)
              {
                usage << "-" << opt->short_opt 
                      << type_str << "|";
              }
            usage << "--" << opt->long_opt << "=" << 
              type_str << " ";
            break;
          default: 
            fprintf (stderr, "gengetopt: bug found in %s:%d!!\n",
                     __FILE__, __LINE__);
            abort ();
          }
      foropt
	if (!opt->required)
          switch (opt->type) {
          case ARG_NO:
          case ARG_FLAG: 
            usage << "[";
            if (opt->short_opt)
              {
                usage << "-" << opt->short_opt << "|";
              }
            usage << "--" << opt->long_opt << "] ";
            break;
          case ARG_INT:
          case ARG_SHORT:
          case ARG_LONG:
          case ARG_FLOAT:
          case ARG_DOUBLE:
          case ARG_LONGDOUBLE:
          case ARG_LONGLONG:
          case ARG_STRING: 
            if (opt->type_str)
              type_str = opt->type_str;
            else
              type_str = arg_names[opt->type];

            if (opt->short_opt)
              {
                usage << "-" << opt->short_opt 
                      << type_str << "|";
              }
            usage << "--" << opt->long_opt << "=" << 
              type_str << " ";
            break;
          default: fprintf (stderr, "gengetopt: bug found in %s:%d!!\n",
                            __FILE__, __LINE__);
            abort ();
          }
    } 
  else 
    { /* if not long help we generate it as GNU standards */
      usage << "[OPTIONS]...";
    }

  if ( unamed_options )
    usage << " [FILES]...";

  usage << "\\n";

  return usage.str ();
}

inline static bool
char_is_newline(const char *buf)
{
  return (*buf == '\\' && (buf+1) != 0 && *(buf+1) == 'n');
}

static void 
generate_help_desc_print(ostream &stream, 
			 unsigned int desc_column,
			 const char *descript, const char *defval)
{
  int  next_space = desc_column;
  char next_word[128];
  int  nw_ind = 0;

  string desc_indent (desc_column + 2, ' ');
  string desc = descript;

  if (defval)
    {
      desc += "  (default=";
      desc += defval;
      desc += ")";
    }

  const char * out_buf = desc.c_str();
  bool found_new_line;

  while (*out_buf)
    {
      found_new_line = false;
      // search next whitespace
      while ((*out_buf) && (*out_buf != ' ') && 
	     (*out_buf != '=') && ! char_is_newline(out_buf))
	{
	  next_word[nw_ind++] = *out_buf++;
	  next_space++;
	}
      if (*out_buf)
	{
	  if (char_is_newline (out_buf))
            {
              found_new_line = true;
              next_space = 79;
              out_buf++; // \n is two chars
            }
	  else
            next_word[nw_ind++] = *out_buf;

	  out_buf++;
	}

      next_word[nw_ind] = 0;
      next_space++;

      // wrap line if it's too long
      if (next_space > 79)
	{
          if (found_new_line)
            stream << next_word;
	  stream << "\\n" << desc_indent;
	  next_space = desc_column + 2;
          if (! found_new_line)
            next_space += strlen (next_word);
	}

      // print word
      if (! found_new_line)
        stream << next_word;
      nw_ind = 0;
    }
}

void 
CmdlineParserCreator::generate_help_option_print(ostream &stream, 
                                                 unsigned int indent)
{
  long desc_col;
  struct gengetopt_option * opt;
  string indent_str (indent, ' ');

  int		type_len;
  const char   *type_str;

  /* calculate columns */
  desc_col = 0;
  foropt {
    int width = 2 + 4 + 2;  // ws + "-a, " + ws

    width += strlen (opt->long_opt) + 2;  // "--"

    if ((opt->type != ARG_FLAG) &&
	(opt->type != ARG_NO))
      {
	if (opt->type_str)
	  type_str = opt->type_str;
	else
	  type_str = arg_names[opt->type];
	type_len = strlen(type_str);
	
	width += type_len + 1;	// "="
      }

    if (width > desc_col)
      desc_col = width;
  }

  /* print justified options */
  char *prev_group = 0;
  print_group_gen_class print_group;
  char *curr_section = 0;
  bool first_option = true;

  stream << indent_str;
  stream << "printf(\"\\n\");\n";

  foropt
    {
      if (opt->group_value && 
          (! prev_group || strcmp (opt->group_value, prev_group) != 0))
        {
          stream << indent_str;
          print_group.set_group_name (opt->group_value);
          print_group.set_group_desc (opt->group_desc);
          print_group.generate_print_group (stream, indent);
          stream << endl;

          prev_group = opt->group_value;
        }

      if (opt->section && 
          (!curr_section || strcmp (curr_section, opt->section)))
        {
          curr_section = opt->section;
          stream << indent_str;
          stream << "printf(\"";
          if (! first_option)
            stream << "\\n";
          stream << opt->section << ":";
          stream << "\\n\");\n";
        }

      first_option = false;

      stream << indent_str;
      stream << "printf(\"";
      if (opt->type == ARG_FLAG || opt->type == ARG_NO)
        {
	  const char * def_val = NULL;
	  
          if (opt->short_opt) 
	    stream << "  -" << opt->short_opt << ", ";
          else
	    stream << "      ";

          stream << "--" << opt->long_opt;

	  string desc_indent (desc_col - 8 - strlen(opt->long_opt), ' ');
	  stream << desc_indent;

	  if (opt->type == ARG_FLAG)
	    def_val = opt->flagstat ? "on" : "off";
	  
	  generate_help_desc_print(stream, desc_col, opt->desc, def_val);
          stream << "\\n\");\n";
        }
      else
        {
	  int col = 0;
	  string def_str = "`";
	  const char * def_val = NULL;

	  if (opt->type_str)
	    type_str = opt->type_str;
	  else
	    type_str = arg_names[opt->type];
	  type_len = strlen(type_str);

          if (opt->short_opt)
	    {
	      stream << "  -" << opt->short_opt << ", ";
	      col += 6;
	    }
          else
            {
              stream << "      ";
	      col += 6;
            }

          stream << "--" << opt->long_opt << "=" << type_str;
	  col += 3 + strlen(opt->long_opt) + type_len;
	  
	  string desc_indent (desc_col - col, ' ');
	  stream << desc_indent;

	  if (opt->default_string)
	    {
	      def_str += opt->default_string;
	      def_str += "'";
	      def_val = def_str.c_str();
	    }

	  generate_help_desc_print(stream, desc_col, opt->desc, def_val);
          stream << "\\n\");\n";
        }
    }
}

void
CmdlineParserCreator::generate_strdup(ostream &stream, unsigned int indent)
{
  struct gengetopt_option * opt;
  bool gen_strdup = unamed_options;

  if (! gen_strdup)
    {
      foropt
	if (opt->type == ARG_STRING) {
          gen_strdup = 1;
          break;
	}
    }

  stream << endl;
  if (gen_strdup) 
    {
      gen_strdup_gen_class strdup_gen;

      strdup_gen.set_prefix ("gengetopt");
      strdup_gen.generate_gen_strdup (stream);
    }
}

void 
CmdlineParserCreator::generate_given_init(ostream &stream, 
                                          unsigned int indent)
{
  struct gengetopt_option * opt;
  string indent_str (indent, ' ');

  /* now we initialize "given" fields */
  foropt
    {
      stream << indent_str;
      stream << ARGS_STRUCT << "->" << opt->var_arg << "_given = 0 ;";
      stream << endl;
    }
}

void
CmdlineParserCreator::generate_struct_def(ostream &stream, unsigned int indent)
{
  struct gengetopt_option * opt;
  string indent_str (indent, ' ');
  multiple_opt_list_gen_class multiple_opt;
  bool generated_counter = false;

  /* define linked-list structs for multiple options */
  foropt
    {
      if (opt->multiple)
	{
          if (! generated_counter)
            {
              stream << indent_str;
              stream << "int i;        /* Counter */" << endl;
              generated_counter = true;
            }
          stream << indent_str;
	  multiple_opt.set_type (arg_types[opt->type]);
	  multiple_opt.set_arg_name (opt->var_arg);
	  multiple_opt.generate_multiple_opt_list (stream, indent);
          stream << endl;
	}
    }
}

void
CmdlineParserCreator::generate_multiple_fill_array(ostream &stream, unsigned int indent)
{
  struct gengetopt_option * opt;
  string indent_str (indent, ' ');
  multiple_fill_array_gen_class filler;

  /* copy linked list into the array */
  foropt
    {
      if (opt->multiple)
	{
	  stream << indent_str;
	  filler.set_type (arg_types[opt->type]);
	  filler.set_option_var_name (opt->var_arg);
	  filler.generate_multiple_fill_array (stream, indent);

          if (opt->default_given)
            {
              multiple_fill_array_default_gen_class def_filler;
              string def_value = opt->default_string;

              def_filler.set_type (arg_types[opt->type]);
              def_filler.set_option_var_name (opt->var_arg);
              if (opt->type == ARG_STRING)
                def_value = "gengetopt_strdup(\"" +
                  def_value + "\")";
              def_filler.set_default_value (def_value);
              def_filler.generate_multiple_fill_array_default (stream, indent);
            }

          stream << endl;
	}
    }
}

void
CmdlineParserCreator::generate_clear_arg(ostream &stream, unsigned int indent)
{
  struct gengetopt_option * opt;
  string indent_str (indent, ' ');

  /* now we initialize "given" fields */
  foropt
    {
      if (opt->multiple)
        {
          // since multiple options are stored in arrays, we
          // always initialize it to NULL

          stream << indent_str;

          stream << ARGS_STRUCT << "->" << opt->var_arg 
                 << "_arg = NULL; \\" << endl;
        }
      else if (opt->type == ARG_STRING)
        {
          stream << indent_str;

          if (opt->default_given)
            stream << ARGS_STRUCT << "->" << opt->var_arg 
                   << "_arg = gengetopt_strdup(\"" << opt->default_string 
                   << "\") ;\\";
          else
            stream << ARGS_STRUCT << "->" << opt->var_arg 
                   << "_arg = NULL; \\";

          stream << endl;
        }
      else if (opt->type == ARG_FLAG)
        {
          stream << indent_str;

          stream << ARGS_STRUCT << "->" << opt->var_arg << "_flag";
          stream << " = " << opt->flagstat << ";\\";

          stream << endl;
        }
      else if (opt->type != ARG_NO && opt->default_given)
        {
          stream << indent_str;

          stream << ARGS_STRUCT << "->" << opt->var_arg << "_arg";
          stream << " = " << opt->default_string << " ;\\";

          stream << endl;
        }
    }
}

void
CmdlineParserCreator::generate_init_unamed(ostream &stream,
                                           unsigned int indent)
{
  if ( unamed_options )
    {
      string indent_str (indent, ' ');

      stream << endl;
      stream << indent_str;
      stream << ARGS_STRUCT << "->inputs = NULL;\n";
      stream << indent_str;
      stream << ARGS_STRUCT << "->inputs_num = 0;\n";
    }
}

void
CmdlineParserCreator::generate_long_option_struct(ostream &stream, 
                                                  unsigned int indent)
{
  string indent_str (indent, ' ');
  struct gengetopt_option * opt;

  foropt
    {
      if (opt->short_opt == 'h' || opt->short_opt == 'V')
        continue;

      stream << indent_str;
      
      stream << "{ \"" << opt->long_opt << "\",\t" 
             << (opt->type == ARG_NO || opt->type == ARG_FLAG ? 0 : 1) 
             << ", NULL, ";

      if (opt->short_opt)
        stream << "\'" << opt->short_opt << "\'";
      else 
        stream << "0";

      stream << " }," << endl;
    }
}

string
CmdlineParserCreator::generate_getopt_string()
{
  struct gengetopt_option * opt;
  ostringstream built_getopt_string;

  foropt
    if (opt->short_opt)
      built_getopt_string << opt->short_opt <<
        (opt->type == ARG_NO || opt->type == ARG_FLAG ? "" : ":");

  return built_getopt_string.str ();
}

void
CmdlineParserCreator::generate_handle_help(ostream &stream, 
                                           unsigned int indent)
{
 if (no_handle_help)
   {
     generic_option_gen_class help_gen;
     help_gen.set_long_option (HELP_LONG_OPT);
     help_gen.set_short_option (HELP_SHORT_OPT_STR);
     help_gen.set_option_comment (HELP_OPT_DESCR);
     help_gen.set_option_var_name (HELP_LONG_OPT);
     help_gen.set_gen_exit ("return");
     help_gen.set_package_var_name (c_source_gen_class::package_var_name);

     help_gen.generate_generic_option (stream, indent);
     
     string indent_str (indent, ' ');
     stream << "return 0;";
   }
 else
   {
     handle_help_gen_class help_gen;
     help_gen.set_parser_name (parser_function_name);
     help_gen.generate_handle_help (stream, indent);
   }
}

void
CmdlineParserCreator::generate_handle_version(ostream &stream, 
                                              unsigned int indent)
{
 if (no_handle_version)
   {
     generic_option_gen_class version_gen;
     version_gen.set_long_option (VERSION_LONG_OPT);
     version_gen.set_short_option (VERSION_SHORT_OPT_STR);
     version_gen.set_option_comment (VERSION_OPT_DESCR);
     version_gen.set_option_var_name (VERSION_LONG_OPT);
     version_gen.set_gen_exit ("return");
     version_gen.set_package_var_name (c_source_gen_class::package_var_name);

     version_gen.generate_generic_option (stream, indent);
     
     string indent_str (indent, ' ');
     stream << "return 0;";
   }
 else
   {
     handle_version_gen_class version_gen;
     version_gen.set_parser_name (parser_function_name);
     version_gen.generate_handle_version (stream, indent);
   }
}

void
CmdlineParserCreator::generate_handle_no_short_option(ostream &stream, 
                                                      unsigned int indent)
{
  struct gengetopt_option * opt;
  no_short_option_gen_class opt_gen;
  string indent_str (indent, ' ');
  opt_gen.set_gen_exit (no_handle_error ? "return" : "exit");
  opt_gen.set_package_var_name (c_source_gen_class::package_var_name);
  multiple_option_no_short_gen_class multip_opt_gen;
  bool first = true;

  foropt
    if (! opt->short_opt)
      {
        stream << indent_str;
        ostringstream str_stream;

        if (opt->multiple)
          {
            multip_opt_gen.set_option_comment (opt->desc);
            multip_opt_gen.set_long_option (opt->long_opt);
            multip_opt_gen.set_option_var_name (opt->var_arg);

            do_update_arg (opt, str_stream, indent + 2);
            multip_opt_gen.set_update_arg (str_stream.str ());

            generic_option_group_gen_class group_gen;
            if (opt->group_value)
              {
                str_stream.str ("");
                str_stream << " ";
                group_gen.set_group_var_name 
                  (canonize_name (opt->group_value));
                group_gen.generate_generic_option_group (str_stream, indent);
                multip_opt_gen.set_update_group_count (str_stream.str ());
              }
            else
              multip_opt_gen.set_update_group_count ("");

            multip_opt_gen.generate_multiple_option_no_short
              (stream, indent);
          }
        else
          {
            opt_gen.set_option_comment (opt->desc);
            opt_gen.set_long_option (opt->long_opt);
            opt_gen.set_option_var_name (opt->var_arg);

            do_update_arg (opt, str_stream, indent + 2);
            opt_gen.set_update_arg (str_stream.str ());

            generic_option_group_gen_class group_gen;
            if (opt->group_value)
              {
                str_stream.str ("");
                str_stream << " ";
                group_gen.set_group_var_name 
                  (canonize_name (opt->group_value));
                group_gen.generate_generic_option_group (str_stream, indent);
                opt_gen.set_update_group_count (str_stream.str ());
              }
            else
              opt_gen.set_update_group_count ("");

            opt_gen.generate_no_short_option (stream, indent);
          }

        stream << endl;

        if (first)
          {
            first = false;
            opt_gen.set_gen_else ("else ");
            multip_opt_gen.set_gen_else ("else ");
          }
      }

  stream << endl;
}

void
CmdlineParserCreator::generate_handle_option(ostream &stream, 
                                             unsigned int indent)
{
  struct gengetopt_option * opt;
  generic_option_gen_class option_gen;
  multiple_option_gen_class multi_gen;
  string indent_str (indent, ' ');

  option_gen.set_package_var_name (c_source_gen_class::package_var_name);

  if (no_handle_error)
    option_gen.set_gen_exit ("return");
  else
    option_gen.set_gen_exit ("exit");

  foropt 
    {
      if (opt->short_opt)
        {
          if (opt->short_opt == 'h' || opt->short_opt == 'V')
            continue;

          stream << indent_str;

          string short_opt (1, opt->short_opt);
	  if (opt->multiple)
	    {
	      multi_gen.set_short_option (short_opt);
	      multi_gen.set_option_comment (opt->desc);
	      multi_gen.set_option_var_name (opt->var_arg);

	      multi_gen.generate_multiple_option (stream, indent);
	    }
	  else
	    {
	      option_gen.set_short_option (short_opt);
	      option_gen.set_option_comment (opt->desc);
	      option_gen.set_long_option (opt->long_opt);
	      option_gen.set_option_var_name (opt->var_arg);

	      option_gen.generate_generic_option (stream, indent);
	    }
          generic_option_group_gen_class group_gen;
          if (opt->group_value)
            {
              group_gen.set_group_var_name 
                (canonize_name (opt->group_value));
              group_gen.generate_generic_option_group (stream, indent);
            }

          do_update_arg (opt, stream, indent + 2);

          stream << endl;
          stream << endl;
        }
    }
}

#define GROUP_REQUIRED_COMPARISON "!="
#define GROUP_NOT_REQUIRED_COMPARISON ">"
#define GROUP_REQUIRED_MESSAGE "One"
#define GROUP_NOT_REQUIRED_MESSAGE "At most one"

void
CmdlineParserCreator::generate_handle_group(ostream &stream,
                                            unsigned int indent)
{
  group_option_gen_class opt_gen;
  string indent_str (indent, ' ');
  opt_gen.set_package_var_name (c_source_gen_class::package_var_name);

  groups_collection_t::const_iterator end = gengetopt_groups.end();
  for ( groups_collection_t::const_iterator idx = gengetopt_groups.begin();
        idx != end; ++idx)
    {
      stream << indent_str;
      opt_gen.set_group_name (idx->first);
      opt_gen.set_group_var_name (canonize_name (idx->first));
      if (idx->second.required)
        {
          opt_gen.set_Comparison_rule(GROUP_REQUIRED_COMPARISON);
          opt_gen.set_number_required(GROUP_REQUIRED_MESSAGE);
        }
      else
        {
          opt_gen.set_Comparison_rule(GROUP_NOT_REQUIRED_COMPARISON);
          opt_gen.set_number_required(GROUP_NOT_REQUIRED_MESSAGE);
        }

      opt_gen.generate_group_option (stream, indent);
      stream << endl;
    }
}

void
CmdlineParserCreator::generate_handle_required(ostream &stream,
                                               unsigned int indent)
{
  struct gengetopt_option * opt;
  required_option_gen_class opt_gen;
  opt_gen.set_package_var_name (c_source_gen_class::package_var_name);
  string indent_str (indent, ' ');

  /* write test for required options */
  foropt
    if ( opt->required )
      {
        stream << indent_str;

        ostringstream req_opt;
        req_opt << "'--" << opt->long_opt << "'";
        if (opt->short_opt)
          req_opt << " ('-" << opt->short_opt << "')";

        opt_gen.set_option_var_name (opt->var_arg);
        opt_gen.set_option_descr (req_opt.str ());

        opt_gen.generate_required_option (stream, indent);

        stream << endl;
      }
}

void
CmdlineParserCreator::generate_group_counters(ostream &stream,
                                              unsigned int indent)
{
  group_counter_gen_class counter_gen;
  string indent_str (indent, ' ');

  groups_collection_t::const_iterator end = gengetopt_groups.end();
  for ( groups_collection_t::const_iterator idx = gengetopt_groups.begin();
        idx != end; ++idx)
    {
      stream << indent_str;
      counter_gen.set_group_name (canonize_name (idx->first));
      counter_gen.generate_group_counter (stream, indent);
      stream << endl;
    }
}

int
CmdlineParserCreator::generate_source ()
{
  /* ****************************************************** */
  /* ********************************************** C FILE  */
  /* ****************************************************** */

  set_usage_string (generate_usage_string ());
  set_getopt_string (generate_getopt_string ());

  ofstream *output_file = open_fstream (c_filename);
  generate_c_source (*output_file);
  output_file->close ();
  delete output_file;

  return 0;
}

void
CmdlineParserCreator::generate_handle_unamed(ostream &stream,
                                             unsigned int indent)
{
  if (! unamed_options)
    return;

  stream << string (indent, ' ');
  handle_unamed_gen_class unamed_gen;
  unamed_gen.generate_handle_unamed (stream, indent);
  stream << endl;
}

void
CmdlineParserCreator::generate_conf_parser(ostream &stream,
                                           unsigned int indent)
{
  if (! conf_parser)
    return;

  stream << endl;
  ConfigParserGenerator conf_parser_gen;
  conf_parser_gen.set_parser_name (c_source_gen_class::parser_name);
  conf_parser_gen.set_gen_exit (gen_exit);
  conf_parser_gen.set_package_var_name (c_source_gen_class::package_var_name);
  conf_parser_gen.generate_config_parser_source (stream);
}

/*
  return a copy of the string passed after canonizing it (i.e. '-' and
  '.' are transformed in '_'.
*/
char *
canonize_names (const char *name)
{
  char *pvar;
  char *p;

  pvar = strdup (name);

  for (p = pvar; *p; ++p)
    if (*p == '.' || *p == '-') *p = '_';

  return pvar;
}

// remove the path from the file name
const string
strip_path(const string &s)
{
  string::size_type pos_of_sep;

  pos_of_sep = s.rfind("/");
  if (pos_of_sep == string::npos)
    pos_of_sep = s.rfind("\\"); // try also with DOS path sep

  if (pos_of_sep == string::npos)
    return s; // no path

  return s.substr (pos_of_sep + 1);
}

const string
to_upper(const string &old)
{
  string upper = old;

  for (string::iterator s = upper.begin ();
       s != upper.end (); ++s)
    *s = toupper (*s);

  return upper;
}

const string
canonize_name(const string &old)
{
  string canonized = old;

  for (string::iterator s = canonized.begin ();
       s != canonized.end (); ++s)
    if (*s == '.' || *s == '-' || *s == ' ') *s = '_';

  return canonized;
}

