/* -*- c++ -*- */
/*
 * Copyright 2002 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 *  Copyright 1997 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */


#ifndef _VRAWGN_H_
#define _VRAWGN_H_

#include <VrSigProc.h>
#include <VrCycleCount.h>

#define IA 16807
#define IM 2147483647
#define AM (1.0/IM)
#define IQ 127773
#define IR 2836
#define NTAB 32
#define NDIV (1+(IM-1)/NTAB)
#define EPS 1.2e-7
#define RNMX (1.0-EPS)

template<class iType>
class VrAWGN : public VrSigProc {
protected:
  float maxAmp, psnr;
  float sigma;
  long  seed;
  int	iset;
  float	gset;
  float ran1();
  float gasdev();
public: 
  virtual const char *name() { return "VrAWGN"; }
  virtual int work(VrSampleRange output, void *o[],
		   VrSampleRange inputs[], void *i[]);

  // SNR is in dB, MA is maximum amplitude
  VrAWGN(float snr = 72, float ma = 1024);
  void setSNR(float arg_snr);
};

/*
 * This looks like it returns a uniform random deviate between 0.0 and 1.0
 * It looks similar to code from "Numerical Recipes in C".
 */
template<class iType> float
VrAWGN<iType>::ran1()
{
    int j;
    long k;
    static long iy=0;
    static long iv[NTAB];
    float temp;

    if (seed <= 0 || !iy) {
        if (-seed < 1) seed=1;
        else seed = -seed;
        for (j=NTAB+7;j>=0;j--) {
            k=seed/IQ;
            seed=IA*(seed-k*IQ)-IR*k;
            if (seed < 0) seed += IM;
            if (j < NTAB) iv[j] = seed;
        }
        iy=iv[0];
    }
    k=(seed)/IQ;
    seed=IA*(seed-k*IQ)-IR*k;
    if (seed < 0)
        seed += IM;
    j=iy/NDIV;
    iy=iv[j];
    iv[j] = seed;
    temp=AM * iy;
    if (temp > RNMX)
        temp = RNMX;
    return temp;
}

/*
 * Returns a normally distributed deviate with zero mean and variance SIGMA.
 * Also looks like it's from "Numerical Recipes in C".
 */
template<class iType> float
VrAWGN<iType>::gasdev()
{
    float fac,rsq,v1,v2;

    iset = 1 - iset;
    if  (iset) {
        do {
            v1=2.0*ran1()-1.0;
            v2=2.0*ran1()-1.0;
            rsq=v1*v1+v2*v2;
        } while (rsq >= 1.0 || rsq == 0.0);
        fac= sqrt(-2.0*log(rsq)/rsq) * sigma;
        gset=v1*fac;
        return v2*fac;
    }
    return gset;
}

template<class iType> int
VrAWGN<iType>::work (VrSampleRange output, void *ao[],
		     VrSampleRange inputs[], void *ai[])
{
  iType **i = (iType **)ai;
  iType **o = (iType **)ao;
  int size = output.size;
  while (size-- > 0) {
    iType temp = *i[0] + (iType)gasdev();
    //jca printf ("i[0] %g temp %g sigma %e\n", *i[0], temp, gasdev() );
    i[0]++;
    *o[0]++ = temp;
  }
  return output.size;
}

template<class iType> void
VrAWGN<iType>::setSNR(float arg_snr)
{
  psnr = arg_snr;
  sigma = maxAmp / pow(10,(psnr/(double)20));
  printf ("snr %f ma %f sigma %f\n", psnr, maxAmp, sigma);
}

template<class iType>
VrAWGN<iType>::VrAWGN(float snr, float ma)
  : VrSigProc(1, sizeof(iType), sizeof(iType))
{
  iset = 0;
  maxAmp = ma;
  setSNR (snr);

  /* seeding random number generator with the cycle counter */
  /* seed needs to be negative */
#if defined (__i386__)
  // Need to fix seeding
  seed = 3021;
  /*
  long long tmp = CYCLE_COUNT();
  if (tmp < 0) seed = -tmp;
  else seed = tmp;
  */
#else
  seed = 3021;
#endif
}
#endif
