/* -*- Mode: c++ -*- 
 *
 *  Copyright 1997 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */


#ifndef _VRFMMOD_H_
#define _VRFMMOD_H_

#include <VrInterpolatingSigProc.h>

#define TAP_RANGE         4194304.0
#define TAP_RANGE_EXP          22

template<class iType,class oType> 
class VrFMMod : public VrInterpolatingSigProc<iType,oType> {
protected:
  float center_freq, amplitude, mod_gain, mod_bw;                     
  int in1, in2, in3, in4;
  int *filter_init_ptr;
  int buffer_length, buffer_bits;            
  int max_input_amplitude;
  unsigned int index, mod_mask, mod_incr, center_incr;
  oType* waveform;
  virtual void initialize();
public:   
  virtual void work(int n);  
  VrFMMod(int, int, float, float, float);
};

template<class iType,class oType> void
VrFMMod<iType,oType>::work(int n)
{
  int *filter_ptr, res1, res2, res3, res4;
  int i,j;
  int index_incr;

  oType *outPtr = outputWritePtr();

  for ( i=0; i < n/interp; i++,incReadPtr(1)) {

    in2 = in1; 
    in3 = in2;
    in4 = in3;
    in1 = inputRead();
    filter_ptr = filter_init_ptr;

    for ( j=0; j<interp; j++){
      
      res1 = in1 * *filter_ptr++;
      res2 = in2 * *filter_ptr++;
      res3 = in3 * *filter_ptr++;
      res4 = in4 * *filter_ptr++;

      index_incr = ((res1 + res2 + res3 + res4) >> TAP_RANGE_EXP) + center_incr;
      index += index_incr;
      //outputWrite( waveform[index & mod_mask]);
      *outPtr++=waveform[index & mod_mask]; //use fast local pointer
    }
  }
  incWritePtr(n);
  return;
}

template<class iType,class oType> void
VrFMMod<iType,oType>::initialize() {

  int M;
  float inSampFreq, outSampFreq;

  mod_mask = 0;
  buffer_length = 1;
  for (int i=0; i < buffer_bits; i++){
    mod_mask = (mod_mask << 1 ) + 1;
    buffer_length = buffer_length * 2;
  }

  inSampFreq = getInputSamplingFrequencyN(0);
  outSampFreq = inSampFreq * (float) interp;
  center_incr = (unsigned int) ((float)buffer_length * center_freq / outSampFreq);

  /* allocate and create waveform */

  waveform = new oType[buffer_length];

  for (int i=0; i<buffer_length; i++) {
    waveform[i] = (oType)(amplitude*sin(2*M_PI*(float)i/(float)buffer_length));
  }

  /* allocate and create interpolation filter */
  /* note that filter coefs are reorderd for easier use later in work()*/

  mod_gain = mod_bw *(float)buffer_length / (float)outSampFreq / (float)max_input_amplitude;
  filter_init_ptr = new int[4*interp];
  M = 4*interp-1;

  for (int i=0; i< interp; i++) {

    filter_init_ptr[4*i]   = (int)(mod_gain*TAP_RANGE*
				   (0.54-0.46*cos(2*M_PI*(float)(i)/(float)M)));
    filter_init_ptr[4*i+1] = (int)(mod_gain*TAP_RANGE*
				   (0.54-0.46*cos(2*M_PI*(float)(i+interp)/(float)M)));
    filter_init_ptr[4*i+2] = (int)(mod_gain*TAP_RANGE*
				   (0.54-0.46*cos(2*M_PI*(float)(i+2*interp)/(float)M)));
    filter_init_ptr[4*i+3] = (int)(mod_gain*TAP_RANGE*
				   (0.54-0.46*cos(2*M_PI*(float)(i+3*interp)/(float)M)));

  }

  in1 = 0;
  in2 = 0;
  in3 = 0;
  in4 = 0;

}

/*    Filter constructor:     buf_bits = b;   2^b = size of output sample lookout table
 *                            interp   = interpolation factor (ratio of in/out sample rates)
 *                            fc       = center frequency of output FM signal
 *                            a        = max amplitude of output samples (8-bit output -> use 127)
 *                            b        = bandwidth of output FM signal (in Hz, i.e. 3000)
 *                              (this is the freq deviation from fc due to max_input_amplitude)  
 */


template<class iType,class oType> 
VrFMMod<iType,oType>::VrFMMod(int buf_bits, int i, float fc, float a, float b)
  :VrInterpolatingSigProc<iType, oType>(1, i), center_freq(fc), amplitude(a),mod_bw(b), buffer_bits(buf_bits)
{
  max_input_amplitude = 127;
  setOutputSize (interp);
  setOutputSizeExact (1); //so we can use fast pointers
}

#endif







