#include <qlabel.h>
#include <qlcdnumber.h>
#include <qbuttongroup.h>
#include <qpushbutton.h>
#include <qpalette.h>
#include <qstring.h>
#include <qvalidator.h>
#include <qkeycode.h>
#include <qlayout.h>
#include <qtimer.h>
#include <stdio.h>
#include "AMFMReceiver.h"
#include "FMTuner.h"
#include "AMTuner.h"
#include <VrAR5000Source.h>
#include <VrComplexFIRfilter.h>
#include <VrQuadratureDemod.h>
#include <VrAmplitudeDemod.h>
#include <VrRealFIRfilter.h>
#include <VrAudioSink.h>

const int inputRate = 33120000;
const float inputFreq = 10740000;

const int CFIRdecimate = 230;
const int chanTaps = 250;
const float chanGain = 2.0;

const float FMdemodGain = 2200;
const float AMdemodGain = 10;

const int RFIRdecimate = 9;
const int ifTaps = 20;
const float ifGain = 1.0;

const int quadRate = inputRate / CFIRdecimate;
const int audioRate = quadRate / RFIRdecimate;

AMFMReceiver::AMFMReceiver(VrAR5000Source<char> *source,
			   QWidget *p, const char *name) : 
  QWidget(p, name), 
  source(source),
  savedSF(source->samplingFrequency()),
  savePreset(false), 
  mode(FM),
  volume(1)
{
  source->setSamplingFrequency(inputRate);

  chanFilter = new VrComplexFIRfilter<char>(CFIRdecimate, chanTaps,
					    inputFreq, chanGain);
  FMdemod = new VrQuadratureDemod<float>(volume * FMdemodGain);
  AMdemod = new VrAmplitudeDemod<float>(volume * AMdemodGain, .05);
  ifFilter = new VrRealFIRfilter<float,short>(RFIRdecimate, audioRate/2,
					      ifTaps, ifGain);
  audio = new VrAudioSink<short>();

  CONNECT(audio, ifFilter, audioRate, 16);
  CONNECT(ifFilter, FMdemod, quadRate, 32);
  CONNECT(FMdemod, chanFilter, quadRate, 64);
  CONNECT(chanFilter, source, inputRate, 8);

  audio->setup();

  fmtuner = new FMTuner(this);
  fmtuner->setWrap(true);
  tuner = fmtuner;

  amtuner = new AMTuner(this);
  amtuner->setWrap(true);

  for (int i=0; i<6; i++)
    {
      presetFMChannel[i] = fmtuner->minChannel();
      presetAMChannel[i] = amtuner->minChannel();
    }
  
  presetChannel = presetFMChannel;

  const QColorGroup &c = colorGroup();
  QColorGroup colors(green, black, c.light(), 
		     c.dark(), c.mid(), green, c.base());
  QPalette palette(colors, colors, colors);

  QBoxLayout *layout = new QVBoxLayout(this);
  layout->addSpacing(10);
  QBoxLayout *hlayout = new QHBoxLayout();
  layout->addLayout(hlayout);
  hlayout->addSpacing(10);

  QWidget *vol = new QWidget(this);
  vol->setFixedSize(40, 60);
  QBoxLayout *vlayout = new QVBoxLayout(vol);
  QButton *volup = new QPushButton("+", vol);
  vlayout->addWidget(volup);
  QLabel *vollabel = new QLabel("Volume", vol);
  vollabel->setAlignment(AlignCenter);
  vlayout->addWidget(vollabel);
  QButton *voldown = new QPushButton("-", vol);
  vlayout->addWidget(voldown);
  hlayout->addWidget(vol);
  hlayout->addSpacing(10);

  QWidget *tune = new QWidget(this);
  tune->setFixedSize(40,60);
  QBoxLayout *tlayout = new QVBoxLayout(tune);
  tuneup = new QPushButton("+", tune);
  tuneup->setAutoRepeat(true);
  tlayout->addWidget(tuneup);
  QLabel *tunelabel = new QLabel("Tune", tune);
  tunelabel->setAlignment(AlignCenter);
  tlayout->addWidget(tunelabel);
  tunedown = new QPushButton("-", tune);
  tunedown->setAutoRepeat(true);
  tlayout->addWidget(tunedown);
  hlayout->addWidget(tune);
  hlayout->addSpacing(10);

  LCDFreq = new QLCDNumber(4, this);
  LCDFreq->setFixedSize(180,60);
  LCDFreq->setSegmentStyle(QLCDNumber::Flat);
  LCDFreq->setSmallDecimalPoint(true);
  LCDFreq->setPalette(palette);
  LCDFreq->setFrameStyle(QFrame::Panel | QFrame::Sunken);
  LCDFreq->setLineWidth(3);
  hlayout->addWidget(LCDFreq);
  hlayout->addSpacing(10);
  
  modeLabel = new QLabel("FM", LCDFreq);
  modeLabel->setPalette(palette);
  modeLabel->adjustSize();
  modeLabel->move(5,5);

  QBoxLayout *blayout = new QVBoxLayout();
  hlayout->addLayout(blayout);
  QBoxLayout *blayout2 = new QHBoxLayout();
  blayout->addLayout(blayout2, 1);
  blayout->addSpacing(5);

  modeButton = new QPushButton("Mode", this);
  modeButton->setFixedWidth(40);
  blayout2->addWidget(modeButton);
  blayout2->addStretch(1);

  memButton = new QPushButton("Mem", this);
  memButton->setFixedWidth(40);
  memButton->setToggleButton(true);
  blayout2->addWidget(memButton);

  presets = new QButtonGroup(this);
  presets->setFrameStyle(QFrame::NoFrame);
  presets->setFixedWidth(160);
  QBoxLayout *playout = new QVBoxLayout(presets);
  QBoxLayout *row = new QHBoxLayout();
  playout->addLayout(row);
  QButton *button = new QPushButton("1", presets);
  row->addWidget(button);
  button = new QPushButton("2", presets);
  row->addWidget(button);
  button = new QPushButton("3", presets);
  row->addWidget(button);
  row = new QHBoxLayout();
  playout->addLayout(row);
  button = new QPushButton("4", presets);
  row->addWidget(button);
  button = new QPushButton("5", presets);
  row->addWidget(button);
  button = new QPushButton("6", presets);
  row->addWidget(button);
  blayout->addWidget(presets, 2);
  hlayout->addSpacing(10);

  layout->addSpacing(10);
  layout->activate();
  layout->freeze();

  QTimer *timer = new QTimer(this);

  connect(modeButton, SIGNAL(clicked()), SLOT(changeMode()));
  connect(memButton, SIGNAL(toggled(bool)), SLOT(memToggled(bool)));
  connect(presets, SIGNAL(clicked(int)), SLOT(presetClicked(int)));

  connect(tuneup, SIGNAL(pressed()), tuner, SLOT(nextChannel()));
  connect(tunedown, SIGNAL(pressed()), tuner, SLOT(prevChannel()));
  connect(tuner, SIGNAL(channelChanged(int)), SLOT(channelChanged(int)));

  connect(volup, SIGNAL(pressed()), SLOT(incVolume()));
  connect(voldown, SIGNAL(pressed()), SLOT(decVolume()));

  connect(timer, SIGNAL(timeout()), SLOT(run()));

  updateFreq();

  timer->start(10);
}

AMFMReceiver::~AMFMReceiver()
{
  delete audio;
  delete ifFilter;
  delete AMdemod;
  delete FMdemod;
  delete chanFilter;

  source->setSamplingFrequency(savedSF);
}

void
AMFMReceiver::updateFreq()
{
  source->setRxFrequency(tuner->frequency());

  char frqString[10];
  if (mode == FM)
    sprintf(frqString, "%.1f", tuner->frequency());
  else
    sprintf(frqString, "%4.0f", tuner->frequency() * 1000);
  LCDFreq->display(frqString);
}

void
AMFMReceiver::memToggled(bool on)
{
  savePreset = on;
}

void
AMFMReceiver::presetClicked(int id)
{
  if (savePreset)
    {
      presetChannel[id] = tuner->channel();
      memButton->setOn(false);
    }
  else
    {
      tuner->setChannel(presetChannel[id]);
    }
}

void
AMFMReceiver::changeMode()
{

  tuneup->disconnect(SIGNAL(pressed()), tuner, SLOT(nextChannel()));
  tunedown->disconnect(SIGNAL(pressed()), tuner, SLOT(prevChannel()));
  tuner->disconnect(SIGNAL(channelChanged(int)), this, SLOT(channelChanged(int)));

  if (mode == AM)
    {
      mode = FM;
      modeLabel->setText("FM");
      tuner = fmtuner;
      presetChannel = presetFMChannel;
      DISCONNECTALLINPUTS(ifFilter);
      DISCONNECTALLINPUTS(AMdemod);
      CONNECT(ifFilter, FMdemod, quadRate, 32);
      CONNECT(FMdemod, chanFilter, quadRate, 64);
    }
  else
    {
      mode = AM;
      modeLabel->setText("AM");
      tuner = amtuner;
      presetChannel = presetAMChannel;
      DISCONNECTALLINPUTS(ifFilter);
      DISCONNECTALLINPUTS(FMdemod);
      CONNECT(ifFilter, AMdemod, quadRate, 32);
      CONNECT(AMdemod, chanFilter, quadRate, 64);
    }

  audio->setup();
  
  
  connect(tuneup, SIGNAL(pressed()), tuner, SLOT(nextChannel()));
  connect(tunedown, SIGNAL(pressed()), tuner, SLOT(prevChannel()));
  connect(tuner, SIGNAL(channelChanged(int)), SLOT(channelChanged(int)));
  updateFreq();
}

void
AMFMReceiver::channelChanged(int channel)
{
  updateFreq();
}

void
AMFMReceiver::incVolume()
{
  if (volume < 10)
    {
      volume++;
      FMdemod->setGain(volume * FMdemodGain);
      AMdemod->setGain(volume * AMdemodGain);
    }
}

void
AMFMReceiver::decVolume()
{
  if (volume > 0)
    {
      volume--;
      FMdemod->setGain(volume * FMdemodGain);
      AMdemod->setGain(volume * AMdemodGain);
    }
}

void
AMFMReceiver::run()
{
  audio->start(0.1);
}
