#include "CordlessPhone.h"

#include <VrAR5000Source.h>
#include <VrComplexFIRfilter.h>
#include <VrSquelch.h>
#include <VrQuadratureDemod.h>
#include <VrRealFIRfilter.h>
#include <VrAudioSink.h>

#include <qlayout.h>
#include <qwt_knob.h>
#include <qlabel.h>
#include <qtimer.h>

const float initVolume = 5.0;
const float squelchThresh = 20000;

const float RxFrequency = 49.24;
const float freqOffset = 0.0;

const int inputRate = 33000000;
const float inputFreq = 10720000;

const int CFIRdecimate = 825;
const int chanTaps = 800;
const float chanGain = 2.0;

const float demodGain = 2200;

const int RFIRdecimate = 5;
const int ifTaps = 20;
const float ifGain = 1.0;

const int quadRate = inputRate / CFIRdecimate;
const int audioRate = quadRate / RFIRdecimate;


CordlessPhone::CordlessPhone(VrAR5000Source<char> *source,
		       QWidget *parent, const char *name) :
  QWidget(parent, name),
  source(source)
{
  source->setRxFrequency(RxFrequency);

  chanFilter = new VrComplexFIRfilter<char>(CFIRdecimate, chanTaps, 
					    inputFreq + freqOffset,
					    chanGain);
  squelch = new VrSquelch<complex>(squelchThresh, .01);
  demod = new VrQuadratureDemod<float>(0.0);
  ifFilter = new VrRealFIRfilter<float,short>(RFIRdecimate, audioRate/2,
					      ifTaps, ifGain);
  audio = new VrAudioSink<short>();


  CONNECT(audio, ifFilter, audioRate, 16);
  CONNECT(ifFilter, demod, quadRate, 32);
  CONNECT(demod, squelch, quadRate, 64);
  CONNECT(squelch, chanFilter, quadRate, 64);
  CONNECT(chanFilter, source, inputRate, 8);

  audio->setup();

  QBoxLayout *layout = new QHBoxLayout(this);

  vol = new QwtKnob(this);
  vol->setFixedSize(80, 80);
  vol->setRange(0, 10, 0.1);
  vol->setScaleMaxMajor(0);

  QLabel *volLabel = new QLabel("Vol", vol);
  volLabel->adjustSize();
  volLabel->setAlignment(AlignCenter);
  volLabel->move((vol->width() - volLabel->width()) / 2,
		 vol->height() - volLabel->height());

  layout->addWidget(vol);
  layout->activate();

  QTimer *timer = new QTimer(this);

  connect(vol, SIGNAL(valueChanged(double)), SLOT(setVolume(double)));
  connect(timer, SIGNAL(timeout()), SLOT(run()));
  
  setVolume(initVolume);

  timer->start(10);

}

CordlessPhone::~CordlessPhone()
{
  delete audio;
  delete ifFilter;
  delete demod;
  delete squelch;
  delete chanFilter;  
}

QSize
CordlessPhone::sizeHint() const
{
  return minimumSize();
}

void
CordlessPhone::setVolume(double volume)
{
  vol->moveTo(volume);
  demod->setGain(vol->value() * demodGain);
}

void
CordlessPhone::run()
{
  audio->start(0.1);
}
