/*
 * This file is part of LibEuFin.
 * Copyright (C) 2024 Taler Systems S.A.

 * LibEuFin is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation; either version 3, or
 * (at your option) any later version.

 * LibEuFin is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General
 * Public License for more details.

 * You should have received a copy of the GNU Affero General Public
 * License along with LibEuFin; see the file COPYING.  If not, see
 * <http://www.gnu.org/licenses/>
 */

import tech.libeufin.common.*
import kotlin.test.Test
import kotlin.test.assertEquals
import kotlin.test.assertFails

class TxMetadataTest {
    fun assertFailsMsg(msg: String, lambda: () -> Unit) {
        val failure = assertFails(lambda)
        assertEquals(msg, failure.message)
    }

    @Test
    fun parse() {
        val upper = "4MZT6RS3RVB3B0E2RDMYW0YRA3Y0VPHYV0CYDE6XBB0YMPFXCEG0"
        val upperL = "4MZT6RS3RVB3B0E2RDMYW0YRA3"
        val upperR = "Y0VPHYV0CYDE6XBB0YMPFXCEG0"
        val mixed = "4mzt6RS3rvb3b0e2rdmyw0yra3y0vphyv0cyde6xbb0ympfxceg0"
        val mixedL = "4mzt6rs3rvb3b0e2rdmyw0yra3"
        val mixedR = "y0vphyv0cyde6xbb0ympfxceg0"
        val otherUpper = "TEST6RS3RVB3B0E2RDMYW0YRA3Y0VPHYV0CYDE6XBB0YMPFXCEG0"
        val otherMixed = "test6rRSrvb3b0e2rdmyw0yra3y0vphyv0cyde6xbb0ympfxceg0"
        val key = TalerIncomingMetadata(TalerIncomingType.reserve, EddsaPublicKey(upper))
        
        // Check succeed if upper or mixed
        for (case in sequenceOf(upper, mixed)) {
            for (test in sequenceOf(
                "noise $case noise",
                "$case noise to the right",
                "noise to the left $case",
                "    $case     ",
                "noise\n$case\nnoise",
                "Test+$case"
            )) {
                assertEquals(key, parseIncomingTxMetadata(test))
            }
        }

        // Check succeed if upper or mixed and split
        for ((L, R) in sequenceOf(upperL to upperR, mixedL to mixedR)) {
            for (case in sequenceOf(
                "left $L$R right",
                "left $L-$R right",
                "left $L+$R right",
                "left $L\n$R right",
                "left $L-+\n$R right",
                "left $L - $R right",
                "left $L + $R right",
                "left $L \n $R right",
                "left $L - + \n $R right",
            )) {
                assertEquals(key, parseIncomingTxMetadata(case))
            }
        } 
        
        // Check prefer upper
        for (case in sequenceOf(
            "$mixed $upper",
            "$mixedL-$mixedR $upperL-$upperR"
        )) {
            assertEquals(key, parseIncomingTxMetadata(case))
        }

        // Check failure when multiple keys match
        for (case in sequenceOf(
            "$upper $upper", // Both upper
            "$mixed $mixed" // Both mixed
        )) {
            assertFailsMsg("Found multiple reserve public key") {
                parseIncomingTxMetadata(case)
            }
        }
       
        // Check failure if malformed or missing
        for (case in sequenceOf(
            "does not contain any reserve", // Check fail if none
            upper.substring(0, upper.length-1) // Check fail if missing char
        )) {
            assertFailsMsg("Missing reserve public key") {
                parseIncomingTxMetadata(case)
            }
        }
    }
}